<?php
// webhook.php : admin-first, Persian UI + emojis, cancel button, discount logic, admin CRUD for user_prices/panels/plans/relations
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/zarinpal.php';
$config = require __DIR__ . '/config.php';
$conn = db_connect();

$rawPost = file_get_contents('php://input');
$update = json_decode($rawPost, true);
if (!$update) { http_response_code(200); exit; }

/* ---------- helpers ---------- */
function send_ok_and_exit() {
    http_response_code(200);
    if (function_exists('fastcgi_finish_request')) fastcgi_finish_request();
    exit;
}
function answer_callback($callback_id, $text = '', $show_alert = false) {
    global $config;
    $url = "https://api.telegram.org/bot{$config['telegram_token']}/answerCallbackQuery";
    $data = ['callback_query_id' => $callback_id, 'text' => $text, 'show_alert' => $show_alert];
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_exec($ch); curl_close($ch);
}
function edit_message_text($chat_id, $message_id, $text, $reply_markup = null) {
    global $config;
    $url = "https://api.telegram.org/bot{$config['telegram_token']}/editMessageText";
    $data = ['chat_id' => $chat_id, 'message_id' => $message_id, 'text' => $text, 'parse_mode' => 'HTML'];
    if ($reply_markup) $data['reply_markup'] = json_encode($reply_markup);
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    $res = curl_exec($ch);
    curl_close($ch);
    return $res;
}
function edit_message_replymarkup($chat_id, $message_id, $reply_markup = null) {
    global $config;
    $url = "https://api.telegram.org/bot{$config['telegram_token']}/editMessageReplyMarkup";
    $data = ['chat_id' => $chat_id, 'message_id' => $message_id];
    if ($reply_markup) $data['reply_markup'] = json_encode($reply_markup);
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    $res = curl_exec($ch);
    curl_close($ch);
    return $res;
}

/* ---------- admin state (small files) ---------- */
function is_admin_user($telegram_id) {
    $conn = db_connect();
    $id = intval($telegram_id);
    $stmt = mysqli_prepare($conn, "SELECT user_status FROM user_details WHERE related_telegram_id = ? LIMIT 1");
    if (!$stmt) return false;
    mysqli_stmt_bind_param($stmt, 'i', $id);
    mysqli_stmt_execute($stmt);
    $res = mysqli_stmt_get_result($stmt);
    $row = mysqli_fetch_assoc($res);
    if (!$row) return false;
    return (isset($row['user_status']) && $row['user_status'] === 'admin');
}
function admin_state_file($tgid) {
    return sys_get_temp_dir() . DIRECTORY_SEPARATOR . "admin_state_{$tgid}.json";
}
function admin_state_load($tgid) {
    $f = admin_state_file($tgid);
    if (!file_exists($f)) return null;
    $c = @file_get_contents($f);
    return $c ? json_decode($c, true) : null;
}
function admin_state_save($tgid, $data) {
    $f = admin_state_file($tgid);
    @file_put_contents($f, json_encode($data, JSON_UNESCAPED_UNICODE));
}
function admin_state_clear($tgid) {
    $f = admin_state_file($tgid);
    @unlink($f);
}

/* admin menu */
function send_admin_menu($chat_id) {
    $kb = ['inline_keyboard' => [
        [['text'=>'📥 پرداخت های معلق','callback_data'=>'admin:view_payments']],
        [['text'=>'🔎 جستجوی کاربر','callback_data'=>'admin:search_user']],
        [['text'=>'🧩 مدیریت پنل ها','callback_data'=>'admin:manage_panels']],
        [['text'=>'🌐 مدیریت هاست نیم ها','callback_data'=>'admin:manage_hostnames']],
        [['text'=>'📋 مدیریت پلن ها','callback_data'=>'admin:manage_plans']],
        [['text'=>'🔗 مدیریت روابط','callback_data'=>'admin:manage_relations']],
        [['text'=>'💸 تنظیمات قیمت کاربران','callback_data'=>'admin:manage_user_prices']],
        [['text'=>'👥 فهرست کاربران','callback_data'=>'admin:list_users:1']],
        [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
    ]];
    telegram_send_message($chat_id, "👋 پنل مدیریت — یکی از عملیات زیر را انتخاب کنید:", $kb);
}

/* get discount percent for a user (by user_status) */
function get_user_discount_percent_by_telegram($telegram_id) {
    $conn = db_connect();
    $tg = intval($telegram_id);
    $stmt = mysqli_prepare($conn, "SELECT user_status FROM user_details WHERE related_telegram_id = ? LIMIT 1");
    mysqli_stmt_bind_param($stmt, 'i', $tg); mysqli_stmt_execute($stmt);
    $row = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt));
    if (!$row) return 0.0;
    $status = $row['user_status'];
    $ps = mysqli_prepare($conn, "SELECT discount FROM user_prices WHERE related_user_status = ? ORDER BY id DESC LIMIT 1");
    mysqli_stmt_bind_param($ps, 's', $status);
    mysqli_stmt_execute($ps);
    $p = mysqli_fetch_assoc(mysqli_stmt_get_result($ps));
    if (!$p) return 0.0;
    return floatval($p['discount']);
}

/* normalize zarinpal wrapper (caller should implement zarinpal_create_payment / zarinpal_verify_payment in zarinpal.php) */
function create_payment_normalized($amount, $order_id, $description='') {
    if (!function_exists('zarinpal_create_payment')) return null;
    $res = @zarinpal_create_payment($amount, $order_id, $description);
    if (!is_array($res)) return null;
    $out = [];
    if (isset($res['authority'])) $out['authority'] = $res['authority'];
    if (isset($res['redirect_url'])) $out['link'] = $res['redirect_url'];
    if (isset($res['data']) && is_array($res['data'])) {
        $d = $res['data'];
        if (isset($d['authority'])) $out['authority'] = $out['authority'] ?? $d['authority'];
        if (isset($d['link'])) $out['link'] = $out['link'] ?? $d['link'];
        $out['data'] = $d;
    }
    if (isset($res['decoded']) && is_array($res['decoded'])) {
        $d = $res['decoded'];
        if (isset($d['data']['authority'])) $out['authority'] = $out['authority'] ?? $d['data']['authority'];
        if (isset($d['data']['link'])) $out['link'] = $out['link'] ?? $d['data']['link'];
        $out['decoded'] = $d;
    }
    return $out;
}

/* ---------- parse update ---------- */
$callback = $update['callback_query'] ?? null;
$message = $update['message'] ?? $update['edited_message'] ?? null;

/* ---------- MESSAGE handling (prioritize admin flows) ---------- */
if ($message) {
    $chat_id = $message['chat']['id'];
    $from_id = $message['from']['id'] ?? null;
    $text = trim($message['text'] ?? '');

    if ($from_id) add_user_if_not_exists($from_id);

    // If admin and in a stateful flow, handle that FIRST (fixes the bug where replies were treated as links)
    if ($from_id && is_admin_user($from_id)) {
        $state = admin_state_load($from_id);
        if ($state) {
            $action = $state['action'] ?? null;
            $step = intval($state['step'] ?? 0);
            $data = $state['data'] ?? [];

            // SEARCH USER (reply handling)
            if ($action === 'search_user' && $step === 1) {
                $tg = intval($text);
                admin_state_clear($from_id);
                if (!$tg) { telegram_send_message($from_id, "❗ شناسهٔ تلگرام نامعتبر است. عملیات لغو شد."); send_ok_and_exit(); }
                $stmt = mysqli_prepare($conn, "SELECT * FROM user_details WHERE related_telegram_id = ? LIMIT 1");
                mysqli_stmt_bind_param($stmt,'i',$tg); mysqli_stmt_execute($stmt);
                $row = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt));
                if (!$row) {
                    telegram_send_message($from_id, "🟡 کاربر پیدا نشد.");
                    send_ok_and_exit();
                }
                $txt = "✅ کاربر پیدا شد:\nID: {$row['id']}\nتلگرام: {$row['related_telegram_id']}\nنقش: {$row['user_status']}\nاعتبار: {$row['credit']}";
                $kb = ['inline_keyboard'=>[
                    [['text'=>'👤 تنظیم نقش → customer','callback_data'=>"admin:internal_set_status:{$row['id']}:customer"]],
                    [['text'=>'👥 تنظیم نقش → reseller','callback_data'=>"admin:internal_set_status:{$row['id']}:reseller"]],
                    [['text'=>'⭐ تنظیم نقش → admin','callback_data'=>"admin:internal_set_status:{$row['id']}:admin"]],
                    [['text'=>'💳 تغییر اعتبار','callback_data'=>"admin:internal_change_credit:{$row['id']}"]],
                    [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
                ]];
                telegram_send_message($from_id, $txt, $kb);
                send_ok_and_exit();
            }

            // ADD PANEL (step-by-step)
            if ($action === 'add_panel') {
                if ($step === 1) {
                    $data['url'] = $text;
                    $state['step'] = 2; $state['data'] = $data; admin_state_save($from_id, $state);
                    $kb = ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]];
                    telegram_send_message($from_id, "🧩 مرحله ۲/۴: نام کاربری پنل را ارسال کنید:", $kb);
                    send_ok_and_exit();
                } elseif ($step === 2) {
                    $data['username'] = $text;
                    $state['step'] = 3; $state['data'] = $data; admin_state_save($from_id, $state);
                    $kb = ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]];
                    telegram_send_message($from_id, "🔐 مرحله ۳/۴: رمز عبور پنل را ارسال کنید:", $kb);
                    send_ok_and_exit();
                } elseif ($step === 3) {
                    $data['password'] = $text;
                    $state['step'] = 4; $state['data'] = $data; admin_state_save($from_id, $state);
                    $kb = ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]];
                    telegram_send_message($from_id, "🏷️ مرحله ۴/۴: server_name (یا '-' برای رد کردن) را ارسال کنید:", $kb);
                    send_ok_and_exit();
                } elseif ($step === 4) {
                    $server_name = ($text === '-') ? null : $text;
                    $data['server_name'] = $server_name;
                    $ins = mysqli_prepare($conn, "INSERT INTO services_panels (url, username, password, server_name) VALUES (?, ?, ?, ?)");
                    mysqli_stmt_bind_param($ins, 'ssss', $data['url'], $data['username'], $data['password'], $data['server_name']);
                    if (!mysqli_stmt_execute($ins)) {
                        telegram_send_message($from_id, "❌ خطا در افزودن پنل: " . mysqli_stmt_error($ins));
                        admin_state_clear($from_id);
                        send_ok_and_exit();
                    }
                    $pid = mysqli_insert_id($conn);
                    telegram_send_message($from_id, "✅ پنل افزوده شد (id = {$pid})");
                    admin_state_clear($from_id);
                    send_ok_and_exit();
                }
            }

            // edit_panel_do: actually update a single field for a panel
            if ($action === 'edit_panel_do' && $step === 1) {
                $panel_id = intval($data['panel_id'] ?? 0);
                $field = $data['field'] ?? '';
                $new_value = $text;
                admin_state_clear($from_id);

                // allowlist fields and map to columns
                $allowed = ['url','username','password','server_name'];
                if (!in_array($field, $allowed, true)) {
                    telegram_send_message($from_id, "❗ فیلد نامعتبر برای ویرایش.");
                    send_ok_and_exit();
                }

                // server_name '-' -> NULL
                if ($field === 'server_name' && $new_value === '-') {
                    $upd = mysqli_prepare($conn, "UPDATE services_panels SET server_name = NULL WHERE id = ?");
                    mysqli_stmt_bind_param($upd, 'i', $panel_id);
                } else {
                    $upd = mysqli_prepare($conn, "UPDATE services_panels SET {$field} = ? WHERE id = ?");
                    mysqli_stmt_bind_param($upd, 'si', $new_value, $panel_id);
                }

                if (!$upd) {
                    telegram_send_message($from_id, "❌ خطا در آماده سازی درخواست: " . mysqli_error($conn));
                    send_ok_and_exit();
                }
                if (!mysqli_stmt_execute($upd)) {
                    telegram_send_message($from_id, "❌ خطا در بروزرسانی: " . mysqli_stmt_error($upd));
                    send_ok_and_exit();
                }
                telegram_send_message($from_id, "✅ فیلد {$field} برای پنل {$panel_id} بروزرسانی شد.");
                send_ok_and_exit();
            }
            
            // ADD HOSTNAME (multi-step)
if ($action === 'add_hostname') {
    if ($step === 1) {
        $hostname = trim($text);
        admin_state_clear($from_id);
        if ($hostname === '-' || $hostname === '') {
            telegram_send_message($from_id, "❌ عملیات لغو شد.");
            send_ok_and_exit();
        }
        // basic validation
        if (!filter_var("http://{$hostname}", FILTER_VALIDATE_URL)) {
            // cheap check: must contain a dot and no spaces
            if (strpos($hostname, ' ') !== false || strpos($hostname, '.') === false) {
                telegram_send_message($from_id, "⚠️ نام هاست‌نیم نامعتبر است. لطفاً دوباره تلاش کنید.");
                send_ok_and_exit();
            }
        }
        // if prefill_panel provided
        $prefill = intval($data['prefill_panel'] ?? 0);
        if ($prefill > 0) {
            // insert directly with prefilled panel
            $ins = mysqli_prepare($conn, "INSERT INTO panels_hostname (hostname, related_panel_id, created_at) VALUES (?, ?, NOW())");
            mysqli_stmt_bind_param($ins, 'si', $hostname, $prefill);
            if (!@mysqli_stmt_execute($ins)) {
                $errno = mysqli_stmt_errno($ins);
                if ($errno === 1062) telegram_send_message($from_id, "⚠️ این هاست‌نیم قبلاً ثبت شده.");
                else telegram_send_message($from_id, "❌ خطا در افزودن: " . mysqli_stmt_error($ins));
            } else {
                $hid = mysqli_insert_id($conn);
                telegram_send_message($from_id, "✅ هاست‌نیم افزوده شد (id = {$hid}) و به پنل {$prefill} مرتبط شد.");
            }
            send_ok_and_exit();
        }

        // otherwise ask for panel id to assign
        admin_state_save($from_id, ['action'=>'add_hostname','step'=>2,'data'=>['hostname'=>$hostname]]);
        telegram_send_message($from_id, "🌐 مرحله ۲/۲: شناسهٔ پنل را برای اختصاص وارد کنید (مثلاً 5) یا '0' برای ثبت بدون اختصاص یا '-' برای لغو:", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
        send_ok_and_exit();
    } elseif ($step === 2) {
        $panel_id = trim($text);
        $hostname = $data['hostname'] ?? null;
        admin_state_clear($from_id);
        if ($panel_id === '-' || $hostname === null) {
            telegram_send_message($from_id, "❌ عملیات لغو شد.");
            send_ok_and_exit();
        }
        $pid = intval($panel_id);
        if ($pid !== 0) {
            // verify panel exists
            $chk = mysqli_prepare($conn, "SELECT id FROM services_panels WHERE id = ? LIMIT 1");
            mysqli_stmt_bind_param($chk,'i',$pid); mysqli_stmt_execute($chk);
            $cres = mysqli_stmt_get_result($chk);
            if (!$cres || mysqli_num_rows($cres) === 0) {
                telegram_send_message($from_id, "❗ پنل وارد شده وجود ندارد. عملیات لغو شد.");
                send_ok_and_exit();
            }
        }
        // insert
        $ins = mysqli_prepare($conn, "INSERT INTO panels_hostname (hostname, related_panel_id, created_at) VALUES (?, ?, NOW())");
        mysqli_stmt_bind_param($ins, 'si', $hostname, $pid);
        if (!@mysqli_stmt_execute($ins)) {
            $errno = mysqli_stmt_errno($ins);
            if ($errno === 1062) telegram_send_message($from_id, "⚠️ این هاست‌نیم قبلاً ثبت شده.");
            else telegram_send_message($from_id, "❌ خطا در افزودن: " . mysqli_stmt_error($ins));
            send_ok_and_exit();
        }
        $hid = mysqli_insert_id($conn);
        telegram_send_message($from_id, "✅ هاست‌نیم ثبت شد (id = {$hid}) — مرتبط با پنل: " . ($pid ? $pid : 'هیچ'));
        send_ok_and_exit();
    }
}


            // NEW: ADD PLAN (multi-step) (unchanged)
            if ($action === 'add_plan') {
                if ($step === 1) {
                    $days = intval($text);
                    if ($days <= 0) { telegram_send_message($from_id, "❗ مقدار روز (duration_days) نامعتبر — لغو شد."); admin_state_clear($from_id); send_ok_and_exit(); }
                    $data['duration_days'] = $days;
                    $state['step'] = 2; $state['data'] = $data; admin_state_save($from_id, $state);
                    telegram_send_message($from_id, "💾 مرحله ۲/۳: حجم را به صورت بایت ارسال کنید (مثال برای 10GiB: 10737418240):", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
                    send_ok_and_exit();
                } elseif ($step === 2) {
                    $bytes = intval($text);
                    if ($bytes <= 0) { telegram_send_message($from_id, "❗ مقدار بایت نامعتبر — لغو شد."); admin_state_clear($from_id); send_ok_and_exit(); }
                    $data['amount'] = $bytes;
                    $state['step'] = 3; $state['data'] = $data; admin_state_save($from_id, $state);
                    telegram_send_message($from_id, "💲 مرحله ۳/۳: قیمت این پلن را ارسال کنید (اعداد):", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
                    send_ok_and_exit();
                } elseif ($step === 3) {
                    $price = floatval($text);
                    if ($price <= 0) { telegram_send_message($from_id, "❗ قیمت نامعتبر — لغو شد."); admin_state_clear($from_id); send_ok_and_exit(); }
                    $ins = mysqli_prepare($conn, "INSERT INTO services_plans (duration_days, amount, price) VALUES (?, ?, ?)");
                    mysqli_stmt_bind_param($ins, 'iid', $data['duration_days'], $data['amount'], $price);
                    if (!mysqli_stmt_execute($ins)) telegram_send_message($from_id, "❌ خطا: " . mysqli_stmt_error($ins));
                    else telegram_send_message($from_id, "✅ پلن افزوده شد.");
                    admin_state_clear($from_id);
                    send_ok_and_exit();
                }
            }

            // EDIT PLAN (expects "duration_days|bytes|price")
            if ($action === 'edit_plan') {
                if ($step === 1) {
                    // text should be: duration_days|amount|price
                    $parts_in = explode('|', $text);
                    if (count($parts_in) !== 3) {
                        telegram_send_message($from_id, "❗ فرمت نامعتبر. به صورت: duration_days|amount|price (مثال: 30|10737418240|10000)");
                        admin_state_clear($from_id);
                        send_ok_and_exit();
                    }
                    $dur = intval($parts_in[0]);
                    $amt = intval($parts_in[1]);
                    $pr = floatval($parts_in[2]);
                    $plan_id = intval($state['data']['id'] ?? 0);
                    if ($plan_id <= 0) { telegram_send_message($from_id,'خطا: plan id نامعتبر'); admin_state_clear($from_id); send_ok_and_exit(); }
                    $up = mysqli_prepare($conn, "UPDATE services_plans SET duration_days = ?, amount = ?, price = ? WHERE id = ?");
                    mysqli_stmt_bind_param($up, 'iidi', $dur, $amt, $pr, $plan_id);
                    if (!mysqli_stmt_execute($up)) telegram_send_message($from_id, "❌ خطا: " . mysqli_stmt_error($up));
                    else telegram_send_message($from_id, "✅ پلن ویرایش شد.");
                    admin_state_clear($from_id);
                    send_ok_and_exit();
                }
            }

            // RELATE PLAN - previous text-based flow kept as fallback (two steps: panel_id -> plan_id)
            if ($action === 'relate_plan') {
                if ($step === 1) {
                    $panel_id = intval($text);
                    if ($panel_id <= 0) { telegram_send_message($from_id, "❗ شناسه پنل نامعتبر — لغو شد."); admin_state_clear($from_id); send_ok_and_exit(); }
                    $state['data']['panel_id'] = $panel_id; $state['step']=2; admin_state_save($from_id,$state);
                    $kb = ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]];
                    telegram_send_message($from_id, "📦 مرحله ۲/۲: شناسه پلن را ارسال کنید تا مرتبط شود:", $kb);
                    send_ok_and_exit();
                } elseif ($step === 2) {
                    $plan_id = intval($text);
                    $panel_id = intval($state['data']['panel_id'] ?? 0);
                    if ($plan_id <=0 || $panel_id <= 0) { telegram_send_message($from_id, "❗ مقادیر نامعتبر — لغو شد."); admin_state_clear($from_id); send_ok_and_exit(); }
                    $ins = mysqli_prepare($conn, "INSERT INTO services_relation_plans (related_panel_id, related_plan_id) VALUES (?, ?)");
                    mysqli_stmt_bind_param($ins, 'ii', $panel_id, $plan_id);
                    if (!mysqli_stmt_execute($ins)) {
                        $errno = mysqli_stmt_errno($ins);
                        if ($errno === 1062) telegram_send_message($from_id, "⚠️ این ارتباط از قبل وجود دارد.");
                        else telegram_send_message($from_id, "❌ خطا: " . mysqli_stmt_error($ins));
                    } else telegram_send_message($from_id, "✅ ارتباط ثبت شد: پنل {$panel_id} → پلن {$plan_id}");
                    admin_state_clear($from_id);
                    send_ok_and_exit();
                }
            }

            // MANAGE user_prices (add/edit) - multi-step (unchanged)
            if ($action === 'manage_user_prices_add') {
                if ($step === 1) {
                    $status = $text;
                    $state['data']['related_user_status'] = $status;
                    $state['step'] = 2; admin_state_save($from_id,$state);
                    telegram_send_message($from_id, "درصد تخفیف را به صورت درصد (مثلاً 12.5) وارد کنید:");
                    send_ok_and_exit();
                } elseif ($step === 2) {
                    $discount = floatval($text);
                    $status = $state['data']['related_user_status'] ?? '';
                    if ($status === '') { telegram_send_message($from_id,'خطا: وضعیت نامعتبر'); admin_state_clear($from_id); send_ok_and_exit(); }
                    $ins = mysqli_prepare($conn, "INSERT INTO user_prices (related_user_status, discount, created_at, updated_at) VALUES (?, ?, NOW(), NOW())");
                    mysqli_stmt_bind_param($ins,'sd', $status, $discount);
                    if (!mysqli_stmt_execute($ins)) telegram_send_message($from_id, "❌ خطا: " . mysqli_stmt_error($ins));
                    else telegram_send_message($from_id, "✅ رکورد قیمت برای '{$status}' با تخفیف {$discount}% افزوده شد.");
                    admin_state_clear($from_id);
                    send_ok_and_exit();
                }
            }

            // change credit internal flow triggered earlier
            if ($action === 'change_credit_internal') {
                if ($step === 1) {
                    $amount = floatval($text);
                    $state['data']['amount'] = $amount; $state['step'] = 2; admin_state_save($from_id,$state);
                    telegram_send_message($from_id, "برای تایید، YES را ارسال کنید یا NO برای لغو:");
                    send_ok_and_exit();
                } elseif ($step === 2) {
                    $confirm = strtoupper($text);
                    if ($confirm === 'YES') {
                        $user_id = intval($state['data']['user_id']);
                        $amount = floatval($state['data']['amount']);
                        $up = mysqli_prepare($conn, "UPDATE user_details SET credit = ? WHERE id = ?");
                        mysqli_stmt_bind_param($up, 'di', $amount, $user_id);
                        if (!mysqli_stmt_execute($up)) telegram_send_message($from_id, "❌ خطا: " . mysqli_stmt_error($up));
                        else telegram_send_message($from_id, "✅ اعتبار کاربر بروز شد.");
                        admin_state_clear($from_id);
                        send_ok_and_exit();
                    } else {
                        telegram_send_message($from_id, "لغو شد.");
                        admin_state_clear($from_id);
                        send_ok_and_exit();
                    }
                }
            }

            // any other admin state: clear
            admin_state_clear($from_id);
            telegram_send_message($from_id, "✅ عملیات کامل یا لغو شد.");
            send_ok_and_exit();
        }
    }

    // If not in admin flow -> proceed with user message processing
    // Check if message contains a proxy link (vmess/vless/trojan/ss)
    $identity = extract_identity_from_link($text);
    if ($identity && $identity['type'] !== 'unknown') {
        // create order
        $req_link = mysqli_real_escape_string($conn, $text);
        $panel_id_placeholder = 0;
        $acc_placeholder = '';
        $stmt = mysqli_prepare($conn, "INSERT INTO orders (telegram_id, panel_id, panel_account_id, service_type, requested_link, price, status) VALUES (?,?,?,?,?,0.00,'searching')");
        mysqli_stmt_bind_param($stmt, 'iisss', $chat_id, $panel_id_placeholder, $acc_placeholder, $identity['type'], $req_link);
        mysqli_stmt_execute($stmt);
        $order_id = mysqli_insert_id($conn);

        // write temp file for worker (worker_search.php reads it)
        $tmpf = __DIR__ . "/tmp_order_search_{$order_id}.json";
        @file_put_contents($tmpf, json_encode(['order_id'=>$order_id, 'chat_id'=>$chat_id, 'text'=>$text, 'parsed'=>$identity], JSON_UNESCAPED_UNICODE));

        // spawn background worker
        $php = PHP_BINARY ?: 'php';
        $worker = __DIR__ . '/worker_search.php';
        $cmd = escapeshellcmd($php) . ' ' . escapeshellarg($worker) . ' ' . escapeshellarg($order_id) . ' > /dev/null 2>&1 &';
        @exec($cmd);

        telegram_send_message($chat_id, "🔎 کانفیگ دریافت شد — در حال جستجو در پنل ها هستم. به زودی گزینه ها را ارسال می کنم.");
        send_ok_and_exit();
    }

    // not a proxy link -> help text
    $help = "برای شروع یک لینک <code>vmess://</code>، <code>vless://</code>، <code>trojan://</code> یا <code>ss://</code> ارسال کنید.\n\n ";
    if ($from_id && is_admin_user($from_id)) {
        $kb = ['inline_keyboard' => [[['text'=>'🛠️ باز کردن پنل مدیریت','callback_data'=>'admin:open']]]];
        telegram_send_message($chat_id, $help, $kb);
    } else {
        telegram_send_message($chat_id, $help);
    }
    send_ok_and_exit();
}

/* ---------- CALLBACK handling ---------- */
if ($callback) {
    $cb = $callback;
    $cb_data = $cb['data'] ?? '';
    $cb_from = $cb['from']['id'] ?? null;
    $cb_msg = $cb['message'] ?? null;
    answer_callback($cb['id'], '⏳ در حال پردازش...', false);

    $parts = explode(':', $cb_data);
    $cmd = $parts[0] ?? '';

    // Admin cancel (cancel current admin flow)
    if ($cmd === 'admin' && ($parts[1] ?? '') === 'cancel') {
        if (!is_admin_user($cb_from)) { answer_callback($cb['id'], 'دسترسی ندارید', true); send_ok_and_exit(); }
        admin_state_clear($cb_from);
        answer_callback($cb['id'], '🛑 عملیات لغو شد', false);
        telegram_send_message($cb_from, "🛑 عملیات فعلی لغو شد.");
        send_ok_and_exit();
    }

    // admin top-level actions
    if ($cmd === 'admin') {
        $sub = $parts[1] ?? 'open';
        if (!is_admin_user($cb_from)) { answer_callback($cb['id'], 'دسترسی ندارید', true); send_ok_and_exit(); }

        
        // ------------------ LIST USERS with pagination ------------------
if ($sub === 'list_users') {
    $page = intval($parts[2] ?? 1);
    if ($page < 1) $page = 1;
    $per = 10;
    $offset = ($page - 1) * $per;

    // total count
    $tot_res = mysqli_query($conn, "SELECT COUNT(*) as c FROM user_details");
    $tot_row = mysqli_fetch_assoc($tot_res);
    $total = intval($tot_row['c'] ?? 0);
    $total_pages = max(1, (int)ceil($total / $per));

    $q = mysqli_prepare($conn, "SELECT id, related_telegram_id, user_status, credit, created_at FROM user_details ORDER BY id ASC LIMIT ?, ?");
    mysqli_stmt_bind_param($q, 'ii', $offset, $per);
    mysqli_stmt_execute($q);
    $res = mysqli_stmt_get_result($q);
    $rows = mysqli_fetch_all($res, MYSQLI_ASSOC);

    if (!$rows) {
        telegram_send_message($cb_from, "🟡 هیچ کاربری پیدا نشد.");
        answer_callback($cb['id'], 'empty', false);
        send_ok_and_exit();
    }

    // build keyboard: each user row → view_user callback
    $keyboard = ['inline_keyboard' => []];
    foreach ($rows as $r) {
        $tgid = $r['related_telegram_id'] ?? '';
        $displayName = $tgid ? fetch_telegram_name($tgid) : '';
        $labelName = $displayName ? " — {$displayName}" : "";
        $label = "👤 #{$r['id']} — {$tgid}{$labelName} — {$r['user_status']} — {$r['credit']}";
        $keyboard['inline_keyboard'][] = [['text' => $label, 'callback_data' => "admin:view_user:{$r['id']}:{$page}"]];
    }

    // navigation row (Persian)
    $nav = [];
    if ($page > 1) $nav[] = ['text' => "⬅️ قبلی", 'callback_data' => "admin:list_users:" . ($page - 1)];
    $nav[] = ['text' => "صفحه {$page}/{$total_pages}", 'callback_data' => "admin:list_users:{$page}"];
    if ($page < $total_pages) $nav[] = ['text' => "بعدی ➡️", 'callback_data' => "admin:list_users:" . ($page + 1)];
    $keyboard['inline_keyboard'][] = $nav;

    $keyboard['inline_keyboard'][] = [['text'=>'✖️ بستن','callback_data'=>'admin:close']];

    telegram_send_message($cb_from, "📋 لیست کاربران — صفحه {$page}/{$total_pages}:", $keyboard);
    answer_callback($cb['id'], 'users sent', false);
    send_ok_and_exit();
}
// ------------------ VIEW SINGLE USER and edit buttons ------------------
if ($sub === 'view_user') {
    $userid = intval($parts[2] ?? 0);
    $came_from_page = intval($parts[3] ?? 1);
    if ($userid <= 0) { answer_callback($cb['id'], 'invalid', true); send_ok_and_exit(); }

    $stmt = mysqli_prepare($conn, "SELECT * FROM user_details WHERE id = ? LIMIT 1");
    mysqli_stmt_bind_param($stmt, 'i', $userid);
    mysqli_stmt_execute($stmt);
    $user = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt));
    if (!$user) { telegram_send_message($cb_from, "🟡 کاربر پیدا نشد."); answer_callback($cb['id'], 'not found', false); send_ok_and_exit(); }

    $tgid = $user['related_telegram_id'] ?? '';
    $tg_name = $tgid ? fetch_telegram_name($tgid) : '';
    $tg_display = $tg_name ? "{$tgid} ({$tg_name})" : ($tgid ?: '-');

    $txt = "👤 کاربر #{$user['id']}\n"
         . "تلگرام: {$tg_display}\n"
         . "نقش: {$user['user_status']}\n"
         . "اعتبار: {$user['credit']}\n"
         . "تاریخ ساخت: {$user['created_at']}\n";

    $kb = ['inline_keyboard' => [
        // edit telegram ID button
        [['text'=>'✏️ ویرایش آیدی تلگرام','callback_data'=>"admin:edit_user_telegram:{$userid}:{$came_from_page}"]],
        // role change
        [['text'=>'تعیین نقش → customer','callback_data'=>"admin:internal_set_status:{$userid}:customer"]],
        [['text'=>'تعیین نقش → reseller','callback_data'=>"admin:internal_set_status:{$userid}:reseller"]],
        [['text'=>'تعیین نقش → admin','callback_data'=>"admin:internal_set_status:{$userid}:admin"]],
        // change credit
        [['text'=>'ویرایش اعتبار','callback_data'=>"admin:internal_change_credit:{$userid}"]],
        [['text'=>'⬅️ بازگشت','callback_data'=>"admin:list_users:{$came_from_page}"]],
        [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
    ]];

    telegram_send_message($cb_from, $txt, $kb);
    answer_callback($cb['id'], 'user shown', false);
    send_ok_and_exit();
}
// ------------------ Begin edit Telegram id (stateful) ------------------
if ($sub === 'edit_user_telegram') {
    $userid = intval($parts[2] ?? 0);
    $came_from_page = intval($parts[3] ?? 1);
    if ($userid <= 0) { answer_callback($cb['id'], 'invalid', true); send_ok_and_exit(); }

    // save state so the next text message from admin will be treated as the new value
    admin_state_save($cb_from, ['action'=>'edit_user_telegram','step'=>1,'data'=>['user_id'=>$userid,'from_page'=>$came_from_page]]);

    telegram_send_message(
        $cb_from,
        "✏️ لطفاً آیدی تلگرام جدید را ارسال کنید (فقط عدد) یا '-' برای لغو:",
        ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]
    );

    answer_callback($cb['id'], 'send new value', false);
    send_ok_and_exit();
}
// ------------------ Stateful: apply edited related_telegram_id ------------------
if ($action === 'edit_user_telegram' && $step === 1) {
    $user_id = intval($data['user_id'] ?? 0);
    $new_tg = trim($text);
    admin_state_clear($from_id);

    if ($new_tg === '-' || $new_tg === '') {
        telegram_send_message($from_id, "❌ عملیات لغو شد.");
        send_ok_and_exit();
    }

    // accept digits only
    if (!preg_match('/^\d+$/', $new_tg)) {
        telegram_send_message($from_id, "⚠️ آیدی تلگرام باید فقط عدد باشد.");
        send_ok_and_exit();
    }

    $nt = intval($new_tg);
    $upd = mysqli_prepare($conn, "UPDATE user_details SET related_telegram_id = ?, updated_at = NOW() WHERE id = ?");
    mysqli_stmt_bind_param($upd, 'ii', $nt, $user_id);

    if (!mysqli_stmt_execute($upd)) {
        telegram_send_message($from_id, "❌ خطا در بروزرسانی: " . mysqli_stmt_error($upd));
        send_ok_and_exit();
    }

    telegram_send_message($from_id, "✅ آیدی تلگرام بروزرسانی شد (User ID: {$user_id}).");
    send_ok_and_exit();
}

        // open menu
        if ($sub === 'open' || $sub === 'menu') {
            send_admin_menu($cb_from);
            answer_callback($cb['id'], 'پنل مدیریت باز شد ✅', false);
            send_ok_and_exit();
        }

        // start SEARCH USER flow (callback -> ask admin to send telegram id)
        if ($sub === 'search_user') {
            admin_state_save($cb_from, ['action'=>'search_user','step'=>1,'data'=>[]]);
            telegram_send_message($cb_from, "🔎 جستجوی کاربر — شناسهٔ تلگرام را (فقط عدد) ارسال کنید:", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
            answer_callback($cb['id'], 'شروع جستجو', false);
            send_ok_and_exit();
        }

        // start ADD PANEL flow (callback)
        if ($sub === 'add_panel') {
            admin_state_save($cb_from, ['action'=>'add_panel','step'=>1,'data'=>[]]);
            telegram_send_message($cb_from, "افزودن پنل — URL پنل را ارسال کنید (مثال: https://example.com):", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
            answer_callback($cb['id'], 'شروع افزودن پنل', false);
            send_ok_and_exit();
        }

        // close: should remove inline keyboard and edit message content if possible
        if ($sub === 'close') {
            if ($cb_msg) {
                $chat = $cb_msg['chat']['id'];
                $mid = $cb_msg['message_id'];
                edit_message_text($chat, $mid, "✖️ پیام بسته شد.");
            }
            answer_callback($cb['id'], 'بسته شد', false);
            send_ok_and_exit();
        }

        // view pending payments
        if ($sub === 'view_payments') {
            $stmt = mysqli_prepare($conn, "SELECT p.id,p.order_id,p.payment_provider,p.provider_ref,p.amount,p.status,o.telegram_id FROM payments p LEFT JOIN orders o ON o.id = p.order_id WHERE p.status='requested' ORDER BY p.id DESC LIMIT 40");
            mysqli_stmt_execute($stmt);
            $res = mysqli_stmt_get_result($stmt);
            $rows = mysqli_fetch_all($res, MYSQLI_ASSOC);
            if (!$rows) {
                telegram_send_message($cb_from, "🟢 هیچ پرداخت معلقی وجود ندارد.");
                answer_callback($cb['id'], 'هیچ پرداختی نیست', false);
                send_ok_and_exit();
            }
            foreach ($rows as $r) {
                $text = "💰 پرداخت #{$r['id']}\nسفارش: {$r['order_id']}\nکاربر: {$r['telegram_id']}\nمبلغ: {$r['amount']}\nمرجع: {$r['provider_ref']}\nوضعیت: {$r['status']}";
                $kb = ['inline_keyboard'=>[
                    [['text'=>'✅ علامت گذاری پرداخت و اعمال','callback_data'=>"admin:apply_payment:{$r['id']}"]],
                    [['text'=>'✖️ بستن این پیام','callback_data'=>'admin:close']]
                ]];
                telegram_send_message($cb_from, $text, $kb);
            }
            answer_callback($cb['id'], 'فهرست پرداخت ها ارسال شد', false);
            send_ok_and_exit();
        }

        // apply payment (admin)
        if ($sub === 'apply_payment') {
            $payid = intval($parts[2] ?? 0);
            if ($payid <= 0) { answer_callback($cb['id'], 'نامعتبر', true); send_ok_and_exit(); }
            $stmt = mysqli_prepare($conn, "SELECT * FROM payments WHERE id=? LIMIT 1");
            mysqli_stmt_bind_param($stmt,'i',$payid); mysqli_stmt_execute($stmt);
            $p = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt));
            if (!$p) { answer_callback($cb['id'],'پرداخت پیدا نشد',true); send_ok_and_exit(); }

            $u1 = mysqli_prepare($conn, "UPDATE payments SET status='verified' WHERE id=?");
            mysqli_stmt_bind_param($u1,'i',$payid); mysqli_stmt_execute($u1);
            $u2 = mysqli_prepare($conn, "UPDATE orders SET status='paid' WHERE id=?");
            mysqli_stmt_bind_param($u2,'i',$p['order_id']); mysqli_stmt_execute($u2);

            // notify buyer and admins
            $order_q = mysqli_prepare($conn, "SELECT * FROM orders WHERE id=? LIMIT 1");
            mysqli_stmt_bind_param($order_q,'i',$p['order_id']); mysqli_stmt_execute($order_q);
            $order = mysqli_fetch_assoc(mysqli_stmt_get_result($order_q));
            telegram_send_message($order['telegram_id'], "✅ پرداخت شما ثبت شد و سفارش به وضعیت «پرداخت شده» تغییر کرد. (سفارش #{$order['id']})");
            telegram_send_to_admins("🔔 ادمین پرداخت {$payid} را برای سفارش {$order['id']} ثبت و اعمال کرد. کاربر: {$order['telegram_id']}");
            answer_callback($cb['id'],'پرداخت اعمال شد',false);
            send_ok_and_exit();
        }

        // manage panels: list panels and allow edit/delete and quick relate + add new panel button
        if ($sub === 'manage_panels') {
            $resq = mysqli_query($conn, "SELECT * FROM services_panels ORDER BY id DESC LIMIT 100");
            $rows = mysqli_fetch_all($resq, MYSQLI_ASSOC);
            $kb_add = ['inline_keyboard'=>[[['text'=>'➕ افزودن پنل جدید','callback_data'=>'admin:add_panel']]]];
            if (!$rows) {
                telegram_send_message($cb_from, "هیچ پنلی ثبت نشده.", $kb_add);
                answer_callback($cb['id'],'خالی',false);
                send_ok_and_exit();
            }
            foreach ($rows as $r) {
                $txt = "🖥 پنل #{$r['id']}\nURL: {$r['url']}\nName: " . ($r['server_name'] ?? '-') ;
                $kb = ['inline_keyboard'=>[
                    [['text'=>'✏️ ویرایش','callback_data'=>"admin:edit_panel:{$r['id']}"]],
                    [['text'=>'🔗 مرتبط کردن','callback_data'=>"admin:relate_choose_panel:{$r['id']}"]],
                    [['text'=>'🗑 حذف','callback_data'=>"admin:delete_panel:{$r['id']}"]],
                    [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
                ]];
                telegram_send_message($cb_from, $txt, $kb);
            }
            // quick add button at the end too
            telegram_send_message($cb_from, "برای افزودن پنل جدید دکمه را بزنید:", $kb_add);
            answer_callback($cb['id'],'پنل ها ارسال شد',false);
            send_ok_and_exit();
        }
        
        // ------------------ Manage hostnames (admin) ------------------
if ($sub === 'manage_hostnames') {
    // quick add button
    $kb_add = ['inline_keyboard'=>[[['text'=>'➕ افزودن هاست‌نیم جدید','callback_data'=>'admin:add_hostname']]]];
    telegram_send_message($cb_from, "🌐 مدیریت هاست‌نیم — برای افزودن هاست‌نیم جدید دکمه را بزنید:", $kb_add);

    $rows = mysqli_fetch_all(mysqli_query($conn, "SELECT h.id, h.hostname, h.related_panel_id, p.server_name FROM panels_hostname h LEFT JOIN services_panels p ON p.id = h.related_panel_id ORDER BY h.id DESC LIMIT 500"), MYSQLI_ASSOC);
    if (!$rows) {
        telegram_send_message($cb_from, "🟡 هیچ هاست‌نیمی ثبت نشده.");
        answer_callback($cb['id'],'hostnames empty',false);
        send_ok_and_exit();
    }
    foreach ($rows as $r) {
        $panel_label = $r['related_panel_id'] ? ("پنل {$r['related_panel_id']}" . ($r['server_name'] ? " — {$r['server_name']}" : "")) : "-";
        $txt = "🌐 هاست‌نیم #{$r['id']}\n{$r['hostname']}\nمرتبط با: {$panel_label}";
        $kb = ['inline_keyboard'=>[
            [['text'=>'🗑 حذف','callback_data'=>"admin:delete_hostname:{$r['id']}"]],
            [['text'=>'🔁 تغییر ارتباط','callback_data'=>"admin:assign_hostname_choose:{$r['id']}"]],
            [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
        ]];
        telegram_send_message($cb_from, $txt, $kb);
    }
    answer_callback($cb['id'],'hostnames sent',false);
    send_ok_and_exit();
}

if ($sub === 'manage_hostnames_for_panel') {
    // expects: admin:manage_hostnames_for_panel:<panel_id>
    $panel_id = intval($parts[2] ?? 0);
    if ($panel_id <= 0) { answer_callback($cb['id'],'invalid',true); send_ok_and_exit(); }

    $stmt = mysqli_prepare($conn, "SELECT id, hostname FROM panels_hostname WHERE related_panel_id = ? ORDER BY id DESC");
    mysqli_stmt_bind_param($stmt,'i',$panel_id);
    mysqli_stmt_execute($stmt);
    $rows = mysqli_fetch_all(mysqli_stmt_get_result($stmt), MYSQLI_ASSOC);
    $kb_add = ['inline_keyboard'=>[[['text'=>'➕ افزودن هاست‌نیم به این پنل','callback_data'=>"admin:add_hostname_for_panel:{$panel_id}"]], [['text'=>'✖️ بستن','callback_data'=>'admin:close']]]];

    if (!$rows) {
        telegram_send_message($cb_from, "برای پنل {$panel_id} هیچ هاست‌نیمی ثبت نشده.", $kb_add);
        answer_callback($cb['id'],'empty for panel',false);
        send_ok_and_exit();
    }
    foreach ($rows as $r) {
        $txt = "🌐 #{$r['id']} — {$r['hostname']}";
        $kb = ['inline_keyboard'=>[
            [['text'=>'🗑 حذف','callback_data'=>"admin:delete_hostname:{$r['id']}"]],
            [['text'=>'🔁 تغییر ارتباط','callback_data'=>"admin:assign_hostname_choose:{$r['id']}"]],
            [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
        ]];
        telegram_send_message($cb_from, $txt, $kb);
    }
    telegram_send_message($cb_from, "برای افزودن هاست‌نیم به این پنل دکمه را بزنید:", $kb_add);
    answer_callback($cb['id'],'hostnames for panel sent',false);
    send_ok_and_exit();
}

if ($sub === 'add_hostname') {
    // start stateful: ask hostname
    admin_state_save($cb_from, ['action'=>'add_hostname','step'=>1,'data'=>[]]);
    telegram_send_message($cb_from, "🌐 افزودن هاست‌نیم — لطفاً نام هاست‌نیم را ارسال کنید (مثلاً example.com) یا '-' برای لغو:", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
    answer_callback($cb['id'],'start add hostname',false);
    send_ok_and_exit();
}

if ($sub === 'add_hostname_for_panel') {
    $panel_id = intval($parts[2] ?? 0);
    if ($panel_id <= 0) { answer_callback($cb['id'],'invalid',true); send_ok_and_exit(); }
    admin_state_save($cb_from, ['action'=>'add_hostname','step'=>1,'data'=>['prefill_panel'=>$panel_id]]);
    telegram_send_message($cb_from, "🌐 افزودن هاست‌نیم به پنل {$panel_id} — لطفاً نام هاست‌نیم را ارسال کنید (مثلاً example.com) یا '-' برای لغو:", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
    answer_callback($cb['id'],'start add hostname for panel',false);
    send_ok_and_exit();
}

if ($sub === 'delete_hostname') {
    $hid = intval($parts[2] ?? 0);
    if ($hid <= 0) { answer_callback($cb['id'],'invalid',true); send_ok_and_exit(); }
    $del = mysqli_prepare($conn, "DELETE FROM panels_hostname WHERE id = ?");
    mysqli_stmt_bind_param($del,'i',$hid); mysqli_stmt_execute($del);
    telegram_send_message($cb_from, "🗑 هاست‌نیم {$hid} حذف شد.");
    answer_callback($cb['id'],'deleted',false);
    send_ok_and_exit();
}

if ($sub === 'assign_hostname_choose') {
    // choose a panel to assign this hostname to
    $hid = intval($parts[2] ?? 0);
    if ($hid <= 0) { answer_callback($cb['id'],'invalid',true); send_ok_and_exit(); }
    $resq = mysqli_query($conn, "SELECT id, server_name, url FROM services_panels ORDER BY id ASC LIMIT 200");
    $rows = mysqli_fetch_all($resq, MYSQLI_ASSOC);
    if (!$rows) {
        telegram_send_message($cb_from, "هیچ پنلی ثبت نشده. ابتدا پنل اضافه کنید.", ['inline_keyboard'=>[[['text'=>'➕ افزودن پنل','callback_data'=>'admin:add_panel']]]]);
        answer_callback($cb['id'],'no panels',false);
        send_ok_and_exit();
    }
    $keyboard = ['inline_keyboard'=>[]];
    foreach ($rows as $r) {
        $label = ($r['server_name'] ?? $r['url'] ?? "پنل {$r['id']}");
        $keyboard['inline_keyboard'][] = [['text' => "🔹 {$label}", 'callback_data' => "admin:assign_hostname_do:{$hid}:{$r['id']}"]];
    }
    $keyboard['inline_keyboard'][] = [['text'=>'❌ جدا کردن از هر پنل (set NULL)','callback_data'=>"admin:assign_hostname_do:{$hid}:0"]];
    $keyboard['inline_keyboard'][] = [['text'=>'✖️ لغو','callback_data'=>'admin:cancel']];
    telegram_send_message($cb_from, "یک پنل انتخاب کنید تا هاست‌نیم به آن اختصاص یابد:", $keyboard);
    answer_callback($cb['id'],'choose panel for hostname',false);
    send_ok_and_exit();
}

if ($sub === 'assign_hostname_do') {
    $hid = intval($parts[2] ?? 0);
    $panel_id = intval($parts[3] ?? 0);
    if ($hid <= 0) { answer_callback($cb['id'],'invalid',true); send_ok_and_exit(); }
    if ($panel_id === 0) {
        // set to 0 -> unassign: (we defined related_panel_id NOT NULL; if you prefer NULL, adjust table definition)
        // here we'll delete or set to 0 (but table has NOT NULL). better to set to 0 if allowed.
        $upd = mysqli_prepare($conn, "UPDATE panels_hostname SET related_panel_id = ? WHERE id = ?");
        mysqli_stmt_bind_param($upd,'ii', $panel_id, $hid);
    } else {
        // ensure panel exists
        $chk = mysqli_prepare($conn, "SELECT id FROM services_panels WHERE id = ? LIMIT 1");
        mysqli_stmt_bind_param($chk,'i',$panel_id); mysqli_stmt_execute($chk);
        $cres = mysqli_stmt_get_result($chk);
        if (!$cres || mysqli_num_rows($cres) === 0) {
            telegram_send_message($cb_from, "❗ پنل انتخاب‌شده وجود ندارد.");
            answer_callback($cb['id'],'panel not found',true);
            send_ok_and_exit();
        }
        $upd = mysqli_prepare($conn, "UPDATE panels_hostname SET related_panel_id = ? WHERE id = ?");
        mysqli_stmt_bind_param($upd,'ii', $panel_id, $hid);
    }
    if (!$upd) { telegram_send_message($cb_from, "❌ خطا در آماده‌سازی درخواست: " . mysqli_error($conn)); answer_callback($cb['id'],'db error',true); send_ok_and_exit(); }
    if (!mysqli_stmt_execute($upd)) { telegram_send_message($cb_from, "❌ خطا در به‌روزرسانی: " . mysqli_stmt_error($upd)); answer_callback($cb['id'],'db error exec',true); send_ok_and_exit(); }
    telegram_send_message($cb_from, "✅ اختصاص/تغییر انجام شد.");
    answer_callback($cb['id'],'assigned',false);
    send_ok_and_exit();
}


        if ($sub === 'edit_panel') {
            $panel_id = intval($parts[2] ?? 0);
            if ($panel_id <=0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            // start stateful flow: ask which field
            admin_state_save($cb_from, ['action'=>'edit_panel','step'=>1,'data'=>['panel_id'=>$panel_id]]);
            $kb = ['inline_keyboard'=>[
                [['text'=>'✏️ تغییر URL','callback_data'=>"admin:edit_panel_field:{$panel_id}:url"]],
                [['text'=>'🔐 تغییر username','callback_data'=>"admin:edit_panel_field:{$panel_id}:username"]],
                [['text'=>'🔑 تغییر password','callback_data'=>"admin:edit_panel_field:{$panel_id}:password"]],
                [['text'=>'🏷️ تغییر server_name','callback_data'=>"admin:edit_panel_field:{$panel_id}:server_name"]],
                [['text'=>'✖️ لغو','callback_data'=>'admin:cancel']]
            ]];
            telegram_send_message($cb_from, "⚙️ کدام فیلد را برای پنل {$panel_id} ویرایش می کنید؟", $kb);
            answer_callback($cb['id'],'شروع ویرایش',false);
            send_ok_and_exit();
        }

        if ($sub === 'edit_panel_field') {
            $panel_id = intval($parts[2] ?? 0);
            $field = $parts[3] ?? '';
            if ($panel_id<=0 || !$field) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            admin_state_save($cb_from, ['action'=>'edit_panel_do','step'=>1,'data'=>['panel_id'=>$panel_id,'field'=>$field]]);
            $hint = ($field === 'server_name') ? "مقدار جدید برای {$field} (یا '-' برای حذف) را ارسال کنید:" : "مقدار جدید برای فیلد {$field} (پنل {$panel_id}) را ارسال کنید:";
            telegram_send_message($cb_from, $hint, ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
            answer_callback($cb['id'],'ارسال مقدار',false);
            send_ok_and_exit();
        }

        if ($sub === 'delete_panel') {
            $panel_id = intval($parts[2] ?? 0);
            if ($panel_id<=0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            $del = mysqli_prepare($conn,"DELETE FROM services_panels WHERE id=?");
            mysqli_stmt_bind_param($del,'i',$panel_id); mysqli_stmt_execute($del);
            telegram_send_message($cb_from, "🗑 پنل {$panel_id} حذف شد.");
            answer_callback($cb['id'],'حذف شد',false);
            send_ok_and_exit();
        }

        // manage plans: list and allow add/edit/delete and quick relate
        if ($sub === 'manage_plans') {
            $resq = mysqli_query($conn, "SELECT * FROM services_plans ORDER BY id DESC LIMIT 200");
            $rows = mysqli_fetch_all($resq, MYSQLI_ASSOC);
            $kb2 = ['inline_keyboard'=>[[['text'=>'➕ افزودن پلن جدید','callback_data'=>'admin:add_plan']]]];
            if (!$rows) { telegram_send_message($cb_from, "هیچ پلنی ثبت نشده.", $kb2); answer_callback($cb['id'],'خالی',false); send_ok_and_exit(); }
            foreach ($rows as $r) {
                $txt = "📋 پلن #{$r['id']}\nمدت (روز): {$r['duration_days']}\nحجم: " . format_bytes($r['amount']) . "\nقیمت (تومان): {$r['price']}";
                $kb = ['inline_keyboard'=>[
                    [['text'=>'✏️ ویرایش','callback_data'=>"admin:edit_plan:{$r['id']}"]],
                    [['text'=>'🔗 مرتبط کردن','callback_data'=>"admin:relate_choose_plan_fromplan:{$r['id']}"]],
                    [['text'=>'🗑 حذف','callback_data'=>"admin:delete_plan:{$r['id']}"]],
                    [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
                ]];
                telegram_send_message($cb_from, $txt, $kb);
            }
            // add new quick button
            telegram_send_message($cb_from, "برای افزودن پلن جدید دکمه را بزنید:", $kb2);
            answer_callback($cb['id'],'پلن ها ارسال شد',false);
            send_ok_and_exit();
        }
        if ($sub === 'add_plan') {
            admin_state_save($cb_from, ['action'=>'add_plan','step'=>1,'data'=>[]]);
            telegram_send_message($cb_from, "افزودن پلن — مدت را به صورت روز ارسال کنید (مثال: 30 برای یک ماه):", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
            answer_callback($cb['id'],'شروع افزودن',false);
            send_ok_and_exit();
        }

        if ($sub === 'edit_plan') {
            $id = intval($parts[2] ?? 0);
            if ($id<=0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            admin_state_save($cb_from, ['action'=>'edit_plan','step'=>1,'data'=>['id'=>$id]]);
            telegram_send_message($cb_from, "مقدار جدید را به صورت: <code>duration_days|amount|price</code> ارسال کنید (مثال: 60|21474836480|20000):", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
            answer_callback($cb['id'],'ارسال مقدار',false); send_ok_and_exit();
        }

        if ($sub === 'delete_plan') {
            $id = intval($parts[2] ?? 0);
            if ($id<=0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            $del = mysqli_prepare($conn,"DELETE FROM services_plans WHERE id=?");
            mysqli_stmt_bind_param($del,'i',$id); mysqli_stmt_execute($del);
            telegram_send_message($cb_from, "🗑 پلن {$id} حذف شد.");
            answer_callback($cb['id'],'حذف شد',false); send_ok_and_exit();
        }

        // manage relations (panel <-> plan) - show current relations and allow add/delete
        if ($sub === 'manage_relations') {
            // show add new relation quick button
            $kb_add = ['inline_keyboard'=>[[['text'=>'➕ افزودن ارتباط جدید','callback_data'=>'admin:add_relation']]]];
            telegram_send_message($cb_from, "مدیریت روابط — برای افزودن رابطه جدید دکمه را بزنید:", $kb_add);

            // list plan relations
            $resq = mysqli_query($conn, "SELECT r.id, r.related_panel_id, r.related_plan_id, p.duration_days, p.amount, p.price FROM services_relation_plans r LEFT JOIN services_plans p ON p.id = r.related_plan_id ORDER BY r.id DESC LIMIT 200");
            $rows = mysqli_fetch_all($resq, MYSQLI_ASSOC);
            if ($rows) {
                foreach ($rows as $r) {
                    $txt = "🔗 relation_plan #{$r['id']}\nپنل: {$r['related_panel_id']} → پلن: {$r['related_plan_id']}\nمدت(روز): {$r['duration_days']}\nحجم: " . format_bytes($r['amount']) . "\nقیمت: {$r['price']}";
                    $kb = ['inline_keyboard'=>[
                        [['text'=>'🗑 حذف','callback_data'=>"admin:delete_relation_plan:{$r['id']}"]],
                        [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
                    ]];
                    telegram_send_message($cb_from, $txt, $kb);
                }
            } else {
                telegram_send_message($cb_from, "هیچ ارتباطی بین پنل و پلن پیدا نشد.");
            }
            answer_callback($cb['id'],'روابط ارسال شد',false);
            send_ok_and_exit();
        }
        if ($sub === 'delete_relation_plan') {
            $rid = intval($parts[2] ?? 0);
            if ($rid<=0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            $del = mysqli_prepare($conn,"DELETE FROM services_relation_plans WHERE id=?");
            mysqli_stmt_bind_param($del,'i',$rid); mysqli_stmt_execute($del);
            telegram_send_message($cb_from, "🗑 حذف شد.");
            answer_callback($cb['id'],'حذف شد',false); send_ok_and_exit();
        }

        // admin: start add relation flow -> show panels list
        if ($sub === 'add_relation') {
            $resq = mysqli_query($conn, "SELECT id, server_name, url FROM services_panels ORDER BY id ASC LIMIT 200");
            $rows = mysqli_fetch_all($resq, MYSQLI_ASSOC);
            if (!$rows) {
                telegram_send_message($cb_from, "هیچ پنلی ثبت نشده. ابتدا پنل اضافه کنید.", ['inline_keyboard'=>[[['text'=>'➕ افزودن پنل','callback_data'=>'admin:add_panel']]]]);
                answer_callback($cb['id'],'پنلی نیست',false);
                send_ok_and_exit();
            }
            $keyboard = ['inline_keyboard'=>[]];
            foreach ($rows as $r) {
                $label = ($r['server_name'] ?? $r['url'] ?? "پنل {$r['id']}");
                $keyboard['inline_keyboard'][] = [['text' => "🔹 {$label}", 'callback_data' => "admin:relate_choose_panel:{$r['id']}"]];
            }
            $keyboard['inline_keyboard'][] = [['text'=>'✖️ لغو','callback_data'=>'admin:cancel']];
            telegram_send_message($cb_from, "ابتدا یک پنل انتخاب کنید:", $keyboard);
            answer_callback($cb['id'],'انتخاب پنل',false);
            send_ok_and_exit();
        }

        // admin: panel selected in relation flow -> list plans to pick
        if ($sub === 'relate_choose_panel') {
            $panel_id = intval($parts[2] ?? 0);
            if ($panel_id <= 0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }

            // fetch plans (all plans)
            $resq = mysqli_query($conn, "SELECT id, duration_days, amount, price FROM services_plans ORDER BY duration_days ASC, amount ASC LIMIT 500");
            $rows = mysqli_fetch_all($resq, MYSQLI_ASSOC);
            if (!$rows) {
                telegram_send_message($cb_from, "هیچ پلنی ثبت نشده. ابتدا پلن اضافه کنید.", ['inline_keyboard'=>[[['text'=>'➕ افزودن پلن','callback_data'=>'admin:add_plan']]]]);
                answer_callback($cb['id'],'پلنی نیست',false);
                send_ok_and_exit();
            }
            $keyboard = ['inline_keyboard'=>[]];
            foreach ($rows as $r) {
                $label = "{$r['duration_days']} روز — " . format_bytes($r['amount']) . " — {$r['price']} تومان";
                $keyboard['inline_keyboard'][] = [['text' => $label, 'callback_data' => "admin:relate_choose_plan:{$panel_id}:{$r['id']}"]];
            }
            $keyboard['inline_keyboard'][] = [['text'=>'✖️ لغو','callback_data'=>'admin:cancel']];
            telegram_send_message($cb_from, "پنل #{$panel_id} انتخاب شد — حالا یک پلن انتخاب کنید تا مرتبط شود:", $keyboard);
            answer_callback($cb['id'],'انتخاب پلن',false);
            send_ok_and_exit();
        }

        // admin: plan selected to relate with panel -> create relation (safer, with checks + logging)
        if ($sub === 'relate_choose_plan') {
            $panel_id = intval($parts[2] ?? 0);
            $plan_id = intval($parts[3] ?? 0);
            if ($panel_id <= 0 || $plan_id <= 0) {
                answer_callback($cb['id'], 'نامعتبر', true);
                send_ok_and_exit();
            }

            @file_put_contents('/tmp/webhook_error.log',
                "[" . date('c') . "] relate_choose_plan called: panel={$panel_id} plan={$plan_id} by {$cb_from}\n",
                FILE_APPEND);

            $chk = @mysqli_prepare($conn, "SELECT id FROM services_relation_plans WHERE related_panel_id = ? AND related_plan_id = ? LIMIT 1");
            if (!$chk) {
                $err = mysqli_error($conn);
                @file_put_contents('/tmp/webhook_error.log', "[" . date('c') . "] prepare(check) failed: {$err}\n", FILE_APPEND);
                answer_callback($cb['id'], 'خطا در پایگاه داده (prepare)', true);
                send_ok_and_exit();
            }
            mysqli_stmt_bind_param($chk, 'ii', $panel_id, $plan_id);
            if (!@mysqli_stmt_execute($chk)) {
                $err = mysqli_stmt_error($chk) ?: mysqli_error($conn);
                @file_put_contents('/tmp/webhook_error.log', "[" . date('c') . "] execute(check) failed: {$err}\n", FILE_APPEND);
                answer_callback($cb['id'], 'خطا در پایگاه داده (execute)', true);
                send_ok_and_exit();
            }
            $cres = mysqli_stmt_get_result($chk);
            if ($cres && mysqli_num_rows($cres) > 0) {
                telegram_send_message($cb_from, "⚠️ این ارتباط از قبل وجود دارد: پنل {$panel_id} → پلن {$plan_id}");
                answer_callback($cb['id'], 'ارتباط وجود دارد', false);
                send_ok_and_exit();
            }

            $ins = @mysqli_prepare($conn, "INSERT INTO services_relation_plans (related_panel_id, related_plan_id) VALUES (?, ?)");
            if (!$ins) {
                $err = mysqli_error($conn);
                @file_put_contents('/tmp/webhook_error.log', "[" . date('c') . "] prepare(insert) failed: {$err}\n", FILE_APPEND);
                answer_callback($cb['id'], 'خطا در پایگاه داده (prepare insert)', true);
                send_ok_and_exit();
            }
            mysqli_stmt_bind_param($ins, 'ii', $panel_id, $plan_id);
            $ok = @mysqli_stmt_execute($ins);
            if (!$ok) {
                $err = mysqli_stmt_error($ins) ?: mysqli_error($conn);
                @file_put_contents('/tmp/webhook_error.log', "[" . date('c') . "] execute(insert) failed: {$err}\n", FILE_APPEND);
                $errno = mysqli_stmt_errno($ins);
                if ($errno === 1062) {
                    telegram_send_message($cb_from, "⚠️ این ارتباط از قبل وجود دارد: پنل {$panel_id} → پلن {$plan_id}");
                    answer_callback($cb['id'], 'ارتباط وجود دارد', false);
                    send_ok_and_exit();
                } else {
                    telegram_send_message($cb_from, "❌ ایجاد ارتباط ناموفق بود: " . $err);
                    answer_callback($cb['id'], 'خطا', true);
                    send_ok_and_exit();
                }
            }

            $rid = mysqli_insert_id($conn);
            telegram_send_message($cb_from, "✅ ارتباط جدید ثبت شد: پنل {$panel_id} → پلن {$plan_id} (id = {$rid})");
            answer_callback($cb['id'], 'ثبت شد', false);
            send_ok_and_exit();
        }

        // alternative flow: admin clicked "relate from plan" — choose panel to attach to plan
        if ($sub === 'relate_choose_plan_fromplan') {
            $plan_id = intval($parts[2] ?? 0);
            if ($plan_id <= 0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            // list panels
            $resq = mysqli_query($conn, "SELECT id, server_name, url FROM services_panels ORDER BY id ASC LIMIT 200");
            $rows = mysqli_fetch_all($resq, MYSQLI_ASSOC);
            if (!$rows) {
                telegram_send_message($cb_from, "هیچ پنلی ثبت نشده. ابتدا پنل اضافه کنید.", ['inline_keyboard'=>[[['text'=>'➕ افزودن پنل','callback_data'=>'admin:add_panel']]]]);
                answer_callback($cb['id'],'پنلی نیست',false);
                send_ok_and_exit();
            }
            $keyboard = ['inline_keyboard'=>[]];
            foreach ($rows as $r) {
                $label = ($r['server_name'] ?? $r['url'] ?? "پنل {$r['id']}");
                $keyboard['inline_keyboard'][] = [['text' => "🔹 {$label}", 'callback_data' => "admin:relate_choose_plan_fromplan_do:{$plan_id}:{$r['id']}"]];
            }
            $keyboard['inline_keyboard'][] = [['text'=>'✖️ لغو','callback_data'=>'admin:cancel']];
            telegram_send_message($cb_from, "ابتدا یک پنل انتخاب کنید تا پلن {$plan_id} به آن مرتبط شود:", $keyboard);
            answer_callback($cb['id'],'انتخاب پنل',false);
            send_ok_and_exit();
        }

        // create relation when started from plan list
        if ($sub === 'relate_choose_plan_fromplan_do') {
            $plan_id = intval($parts[2] ?? 0);
            $panel_id = intval($parts[3] ?? 0);
            if ($panel_id <= 0 || $plan_id <= 0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            $ins = mysqli_prepare($conn, "INSERT INTO services_relation_plans (related_panel_id, related_plan_id) VALUES (?, ?)");
            mysqli_stmt_bind_param($ins, 'ii', $panel_id, $plan_id);
            $ok = @mysqli_stmt_execute($ins);
            if (!$ok) {
                $errno = mysqli_stmt_errno($ins);
                if ($errno === 1062) {
                    telegram_send_message($cb_from, "⚠️ این ارتباط از قبل وجود دارد: پنل {$panel_id} → پلن {$plan_id}");
                    answer_callback($cb['id'],'ارتباط وجود دارد',false);
                    send_ok_and_exit();
                } else {
                    telegram_send_message($cb_from, "❌ ایجاد ارتباط ناموفق بود: " . mysqli_stmt_error($ins));
                    answer_callback($cb['id'],'خطا',true);
                    send_ok_and_exit();
                }
            }
            $rid = mysqli_insert_id($conn);
            telegram_send_message($cb_from, "✅ ارتباط جدید ثبت شد: پنل {$panel_id} → پلن {$plan_id} (id = {$rid})");
            answer_callback($cb['id'],'ثبت شد',false);
            send_ok_and_exit();
        }

        // manage user_prices (list/add/edit/delete)
        if ($sub === 'manage_user_prices') {
            $rows = mysqli_fetch_all(mysqli_query($conn, "SELECT * FROM user_prices ORDER BY id DESC LIMIT 200"), MYSQLI_ASSOC);
            if (!$rows) { telegram_send_message($cb_from, "هیچ قیمتی تنظیم نشده."); answer_callback($cb['id'],'خالی',false); send_ok_and_exit(); }
            foreach ($rows as $r) {
                $txt = "💸 user_prices #{$r['id']}\nstatus: {$r['related_user_status']}\ndiscount: {$r['discount']}%";
                $kb = ['inline_keyboard'=>[
                    [['text'=>'✏️ ویرایش','callback_data'=>"admin:edit_user_price:{$r['id']}"]],
                    [['text'=>'🗑 حذف','callback_data'=>"admin:delete_user_price:{$r['id']}"]],
                    [['text'=>'✖️ بستن','callback_data'=>'admin:close']]
                ]];
                telegram_send_message($cb_from, $txt, $kb);
            }
            // add new quick button
            $kb2 = ['inline_keyboard'=>[[['text'=>'➕ افزودن قیمت جدید','callback_data'=>'admin:add_user_price']]]];
            telegram_send_message($cb_from, "برای افزودن قیمت جدید دکمه را بزنید:", $kb2);
            answer_callback($cb['id'],'user_prices ارسال شد',false);
            send_ok_and_exit();
        }

        if ($sub === 'add_user_price') {
            admin_state_save($cb_from, ['action'=>'manage_user_prices_add','step'=>1,'data'=>[]]);
            telegram_send_message($cb_from, "افزودن قیمت — وضعیت کاربر را ارسال کنید (مثلاً customer یا reseller یا admin):", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
            answer_callback($cb['id'],'شروع افزودن',false);
            send_ok_and_exit();
        }

        if ($sub === 'edit_user_price') {
            $id = intval($parts[2] ?? 0);
            if ($id<=0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            admin_state_save($cb_from, ['action'=>'edit_user_price','step'=>1,'data'=>['id'=>$id]]);
            telegram_send_message($cb_from, "مقدار جدید را به صورت: <code>related_user_status|discount</code> ارسال کنید (مثال: reseller|12.5):", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
            answer_callback($cb['id'],'ارسال مقدار',false); send_ok_and_exit();
        }

        if ($sub === 'delete_user_price') {
            $id = intval($parts[2] ?? 0);
            if ($id<=0) { answer_callback($cb['id'],'نامعتبر',true); send_ok_and_exit(); }
            $del = mysqli_prepare($conn,"DELETE FROM user_prices WHERE id=?");
            mysqli_stmt_bind_param($del,'i',$id); mysqli_stmt_execute($del);
            telegram_send_message($cb_from, "🗑 رکورد {$id} حذف شد.");
            answer_callback($cb['id'],'حذف شد',false); send_ok_and_exit();
        }

        // internal set status / change credit callbacks handled previously with internal_ prefix
        if ($sub === 'internal_set_status') {
            $userid = intval($parts[2] ?? 0);
            $newstatus = $parts[3] ?? '';
            if ($userid && in_array($newstatus, ['customer','reseller','admin'])) {
                $u = mysqli_prepare($conn, "UPDATE user_details SET user_status = ? WHERE id = ?");
                mysqli_stmt_bind_param($u,'si',$newstatus, $userid); mysqli_stmt_execute($u);
                telegram_send_message($cb_from, "✅ نقش کاربر تغییر کرد به '{$newstatus}'.");
                answer_callback($cb['id'], 'تغییر یافت', false);
                send_ok_and_exit();
            }
        }
        if ($sub === 'internal_change_credit') {
            $userid = intval($parts[2] ?? 0);
            if ($userid) {
                admin_state_save($cb_from, ['action'=>'change_credit_internal','step'=>1,'data'=>['user_id'=>$userid]]);
                telegram_send_message($cb_from, "مقدار جدید اعتبار را ارسال کنید (اعداد):", ['inline_keyboard'=>[[['text'=>'🔙 لغو','callback_data'=>'admin:cancel']]]]);
                answer_callback($cb['id'], 'ارسال مقدار', false);
                send_ok_and_exit();
            }
        }

        // fallback admin
        answer_callback($cb['id'], 'دستور مدیریت نامشخص است', true);
        send_ok_and_exit();
    } // end admin

    /* ---------- user callbacks ---------- */

    // close generic: close the message's keyboard and edit text to "closed"
    if ($cmd === 'close') {
        if ($cb_msg) {
            $chat = $cb_msg['chat']['id'];
            $mid = $cb_msg['message_id'];
            edit_message_text($chat, $mid, "✖️ پیام بسته شد.");
        }
        answer_callback($cb['id'], 'بسته شد', false);
        send_ok_and_exit();
    }

    // pick_d (duration) -> now shows available PLANS (data amounts) for chosen duration
    // expected callback: pick_d:<order_id>:<duration_days>
    if ($cmd === 'pick_d') {
        $order_id = intval($parts[1] ?? 0);
        $duration_days = intval($parts[2] ?? 0);
        if ($order_id <= 0 || $duration_days <= 0) { answer_callback($cb['id'], 'انتخاب نامعتبر', true); send_ok_and_exit(); }

        $q = mysqli_prepare($conn, "SELECT * FROM orders WHERE id=? LIMIT 1");
        mysqli_stmt_bind_param($q, 'i', $order_id); mysqli_stmt_execute($q);
        $res = mysqli_stmt_get_result($q);
        $order = mysqli_fetch_assoc($res);
        if (!$order) { answer_callback($cb['id'], 'سفارش یافت نشد', true); send_ok_and_exit(); }
        if (intval($order['telegram_id']) !== intval($cb_from)) { answer_callback($cb['id'], 'این سفارش شما نیست', true); send_ok_and_exit(); }

        // find plans for this panel and chosen duration
        $panel_id = intval($order['panel_id']);
        $stmt = mysqli_prepare($conn, "SELECT p.id, p.amount, p.price FROM services_plans p JOIN services_relation_plans r ON p.id = r.related_plan_id WHERE r.related_panel_id = ? AND p.duration_days = ? ORDER BY p.amount ASC");
        mysqli_stmt_bind_param($stmt, 'ii', $panel_id, $duration_days);
        mysqli_stmt_execute($stmt);
        $dres = mysqli_stmt_get_result($stmt);
        $plans = mysqli_fetch_all($dres, MYSQLI_ASSOC);
        if (!$plans) {
            edit_message_text($cb_msg['chat']['id'], $cb_msg['message_id'], "⚠️ برای مدت انتخاب شده ({$duration_days} روز) هیچ پلنی در این پنل تعریف نشده است. لطفاً مدت دیگری انتخاب کنید.", ['inline_keyboard'=>[[['text'=>'✖️ انصراف','callback_data'=>"cancel_order:{$order_id}"]]]]);
            answer_callback($cb['id'], 'پلنی برای این مدت موجود نیست', false);
            send_ok_and_exit();
        }
        $keyboard = ['inline_keyboard'=>[]];
        foreach ($plans as $p) {
            $label = format_bytes($p['amount']) . ' — ' . $p['price'] . ' تومان ';
            $keyboard['inline_keyboard'][] = [['text'=>$label, 'callback_data'=>"pick_s:{$order_id}:{$p['id']}"]];
        }
        $keyboard['inline_keyboard'][] = [['text'=>'✖️ انصراف','callback_data'=>"cancel_order:{$order_id}"]];
        edit_message_text($cb_msg['chat']['id'], $cb_msg['message_id'], "✅ مدت انتخاب شد. لطفاً یک پکیج (حجم) را انتخاب کنید:", $keyboard);
        answer_callback($cb['id'], 'مدت ثبت شد', false);
        send_ok_and_exit();
    }
    

    // pick_s (plan) -> create payment with plan price applied
    // expected callback: pick_s:<order_id>:<plan_id>
if ($cmd === 'pick_s') {
    // --- logging helper ---
    function write_log($msg) {
        $dir = __DIR__ . '/logs';
        if (!is_dir($dir)) @mkdir($dir, 0755, true);
        $file = $dir . '/payments.log';
        $entry = "[" . date('c') . "] " . $msg . PHP_EOL;
        @file_put_contents($file, $entry, FILE_APPEND | LOCK_EX);
    }
    function log_var($v) {
        $j = @json_encode($v, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        if ($j === false) return print_r($v, true);
        return $j;
    }

    $order_id = intval($parts[1] ?? 0);
    $plan_id  = intval($parts[2] ?? 0);
    write_log("pick_s called; raw parts: " . log_var($parts) . " -> order_id={$order_id}, plan_id={$plan_id}, cb_from={$cb_from}");
    if ($order_id <= 0 || $plan_id <= 0) { write_log("invalid params"); answer_callback($cb['id'],'انتخاب نامعتبر',true); send_ok_and_exit(); }

    // --- load order safely ---
    $q = mysqli_prepare($conn, "SELECT id, telegram_id FROM orders WHERE id=? LIMIT 1");
    if (!$q) { write_log("prepare orders failed: " . mysqli_error($conn)); answer_callback($cb['id'],'خطا در سرور (prepare orders)',true); send_ok_and_exit(); }
    mysqli_stmt_bind_param($q, 'i', $order_id);
    if (!mysqli_stmt_execute($q)) { write_log("exec orders failed: " . mysqli_error($conn)); answer_callback($cb['id'],'خطا در سرور (exec orders)',true); send_ok_and_exit(); }
    mysqli_stmt_bind_result($q, $ord_id_res, $ord_telegram_id_res);
    $order = null;
    if (mysqli_stmt_fetch($q)) {
        $order = ['id' => $ord_id_res, 'telegram_id' => $ord_telegram_id_res];
    }
    mysqli_stmt_close($q);

    if (!$order) { write_log("order not found: {$order_id}"); answer_callback($cb['id'],'سفارش یافت نشد',true); send_ok_and_exit(); }
    if (intval($order['telegram_id']) !== intval($cb_from)) { write_log("order ownership mismatch: order.tg={$order['telegram_id']} cb_from={$cb_from}"); answer_callback($cb['id'],'این سفارش شما نیست',true); send_ok_and_exit(); }

    // --- fetch plan row ---
    $pstmt = mysqli_prepare($conn, "SELECT id, price FROM services_plans WHERE id = ? LIMIT 1");
    if (!$pstmt) { write_log("prepare plans failed: " . mysqli_error($conn)); answer_callback($cb['id'],'خطا در سرور (prepare plans)',true); send_ok_and_exit(); }
    mysqli_stmt_bind_param($pstmt, 'i', $plan_id);
    if (!mysqli_stmt_execute($pstmt)) { write_log("exec plans failed: " . mysqli_error($conn)); mysqli_stmt_close($pstmt); answer_callback($cb['id'],'خطا در سرور (exec plans)',true); send_ok_and_exit(); }
    mysqli_stmt_bind_result($pstmt, $plan_id_res, $plan_price_res);
    $plan = null;
    if (mysqli_stmt_fetch($pstmt)) {
        $plan = ['id' => $plan_id_res, 'price' => $plan_price_res];
    }
    mysqli_stmt_close($pstmt);

    if (!$plan) { write_log("plan not found: {$plan_id}"); answer_callback($cb['id'],'پلن یافت نشد',true); send_ok_and_exit(); }

    $base_total = floatval($plan['price']);
    write_log("plan loaded: id={$plan['id']}, price={$base_total}");

    // --- discount logic ---
    $discount_percent = get_user_discount_percent_by_telegram($order['telegram_id']);
    $discount_text = '';
    $final_total = $base_total;
    if ($discount_percent > 0.0) {
        $final_total = round($base_total * (1.0 - ($discount_percent / 100.0)), 2);
        $discount_text = " (تخفیف {$discount_percent}% اعمال شد)";
    }
    write_log("discount computed: percent={$discount_percent}, final_total={$final_total}");

    // --- update order with selected plan and price ---
    $u = mysqli_prepare($conn, "UPDATE orders SET plan_id = ?, price = ? WHERE id = ?");
    if (!$u) { write_log("prepare update failed: " . mysqli_error($conn)); answer_callback($cb['id'],'خطا در سرور (prepare update)',true); send_ok_and_exit(); }
    mysqli_stmt_bind_param($u, 'idi', $plan_id, $final_total, $order_id);
    if (!mysqli_stmt_execute($u)) { write_log("exec update failed: " . mysqli_error($conn)); mysqli_stmt_close($u); answer_callback($cb['id'],'خطا در سرور (update)',true); send_ok_and_exit(); }
    mysqli_stmt_close($u);
    write_log("order updated: order_id={$order_id}, plan_id={$plan_id}, price={$final_total}");

    // --- create payment (wrapper) ---
    write_log("calling create_payment_normalized(amount={$final_total}, order_id={$order_id})");
    $z = create_payment_normalized($final_total, $order_id, "سفارش #{$order_id}");
    write_log("create_payment_normalized response: " . log_var($z));
    $authority = null;
    $pay_link  = null;

    // normalize possible response shapes
    if (is_array($z)) {
        if (!empty($z['authority'])) {
            $authority = $z['authority'];
        } elseif (!empty($z['data']['authority'])) {
            $authority = $z['data']['authority'];
        } elseif (!empty($z['data']['url'])) {
            $pay_link = $z['data']['url'];
        } elseif (!empty($z['link'])) {
            $pay_link = $z['link'];
        }
    } else {
        write_log("create_payment_normalized returned non-array: " . var_export($z, true));
    }

    // fallback: if authority present but pay_link missing, build StartPay URL (sandbox vs prod)
    $isSandbox = !empty($config['zarinpal_sandbox']);
    if ($authority && !$pay_link) {
        $startHost = $isSandbox ? 'https://sandbox.zarinpal.com/pg/StartPay/' : 'https://www.zarinpal.com/pg/StartPay/';
        $pay_link = $startHost . rawurlencode($authority);
    }

    if (!$authority && !$pay_link) {
        write_log("no authority and no pay_link; aborting. z_response=" . log_var($z));
        answer_callback($cb['id'],'ایجاد پرداخت ناموفق بود',true); send_ok_and_exit();
    }

    // --- insert payment row ---
    $ins = mysqli_prepare($conn, "INSERT INTO payments (order_id, payment_provider, provider_ref, amount, status) VALUES (?,?,?,?, 'requested')");
    if ($ins) {
        $provider = 'zarinpal';
        mysqli_stmt_bind_param($ins, 'issd', $order_id, $provider, $authority, $final_total);
        if (!mysqli_stmt_execute($ins)) {
            write_log("payments insert failed: " . mysqli_error($conn) . " | order_id={$order_id} authority={$authority} amount={$final_total}");
        } else {
            write_log("payments insert succeeded: order_id={$order_id} authority={$authority} amount={$final_total}");
        }
        mysqli_stmt_close($ins);
    } else {
        write_log("prepare payments insert failed: " . mysqli_error($conn));
        // continue, not fatal
    }

    // --- build signed redirector link (preferred) ---
    $site = rtrim($config['site_url'] ?? ($config['domain'] ?? ''), '/');
    $secret = $config['z_sign_key'] ?? '';

    if ($site && $secret && $authority) {
        $ts = time();
        $order_for_sig = (int)$order_id;
        $sig = hash_hmac('sha256', "{$authority}|{$order_for_sig}|{$ts}", $secret);
        $pay_link = $site . "/pay_redirect/?auth=" . rawurlencode($authority)
                  . "&order=" . rawurlencode($order_for_sig)
                  . "&ts=" . rawurlencode($ts)
                  . "&sig=" . rawurlencode($sig);
        write_log("signed redirector built: {$pay_link}");
    } else {
        $startHost = $isSandbox ? 'https://sandbox.zarinpal.com/pg/StartPay/' : 'https://www.zarinpal.com/pg/StartPay/';
        if ($authority) {
            $pay_link = $startHost . rawurlencode($authority);
            write_log("using direct StartPay link: {$pay_link}");
        } elseif ($pay_link) {
            write_log("using wrapper-provided pay_link: {$pay_link}");
        } else {
            write_log("unable to build any pay link for order {$order_id}");
            answer_callback($cb['id'],'فعلاً امکان ساخت لینک پرداخت وجود ندارد',true);
            send_ok_and_exit();
        }
    }

    // --- keyboard + message ---
    $keyboard = ['inline_keyboard'=>[
        [['text'=>'💳 پرداخت (رفتن به درگاه)','url'=>$pay_link]],
        [['text'=>'✖️ انصراف','callback_data'=>"cancel_order:{$order_id}"]]
    ]];

    $display_total = number_format((float)$final_total, 2, '.', ','); // nice formatting
    $msg_text = "💳 پرداخت ایجاد شد.\nمبلغ نهایی: {$display_total} تومان{$discount_text}\nبرای پرداخت روی دکمه بزنید.";
    write_log("finalizing: order_id={$order_id}, user={$order['telegram_id']}, display_total={$display_total}, pay_link={$pay_link}");
    edit_message_text($cb_msg['chat']['id'], $cb_msg['message_id'], $msg_text, $keyboard);

    telegram_send_to_admins("🔔 پرداخت درخواستی: سفارش {$order_id} — کاربر {$order['telegram_id']} — مبلغ: {$display_total}");
    answer_callback($cb['id'], 'پرداخت ایجاد شد ✅', false);
    write_log("pick_s finished successfully for order {$order_id}");
    send_ok_and_exit();
}


    // verify_payment (if user clicks verify maybe)
    if ($cmd === 'verify_payment') {
        $order_id = intval($parts[1] ?? 0);
        $authority = $parts[2] ?? null;
        if ($order_id <=0 || !$authority) { answer_callback($cb['id'], 'نامعتبر', true); send_ok_and_exit(); }
        $ver = null;
        if (function_exists('zarinpal_verify_payment')) {
            $amt = 0.0;
            $pstmt = mysqli_prepare($conn, "SELECT amount FROM payments WHERE provider_ref=? LIMIT 1");
            mysqli_stmt_bind_param($pstmt, 's', $authority); mysqli_stmt_execute($pstmt);
            $prow = mysqli_fetch_assoc(mysqli_stmt_get_result($pstmt));
            if ($prow) $amt = floatval($prow['amount']);
            $ver = @zarinpal_verify_payment($authority, $amt);
        }
        $ok = false;
        if (is_array($ver)) {
            if (isset($ver['data']['code']) && intval($ver['data']['code']) === 100) $ok = true;
            if (!empty($ver['ref_id']) || !empty($ver['data']['ref_id'])) $ok = true;
            if (!empty($ver['status']) && $ver['status'] === 'success') $ok = true;
        }
        if ($ok) {
            // mark payment & order as paid
            $stmt = mysqli_prepare($conn, "UPDATE payments SET status='verified' WHERE provider_ref=?");
            mysqli_stmt_bind_param($stmt, 's', $authority); mysqli_stmt_execute($stmt);
            $u2 = mysqli_prepare($conn, "UPDATE orders SET status='paid' WHERE id=?");
            mysqli_stmt_bind_param($u2, 'i', $order_id); mysqli_stmt_execute($u2);
            // notify and attempt apply (same logic as before)...
            telegram_send_to_admins("✅ پرداخت تایید شد: سفارش {$order_id}");
            answer_callback($cb['id'], 'پرداخت تایید شد ✅', false);
        } else {
            answer_callback($cb['id'], 'پرداخت هنوز تایید نشده یا خطا', true);
        }
        send_ok_and_exit();
    }

    // cancel order
    if ($cmd === 'cancel_order') {
        $order_id = intval($parts[1] ?? 0);
        if ($order_id <=0) { answer_callback($cb['id'], 'نامعتبر', true); send_ok_and_exit(); }
        $q = mysqli_prepare($conn, "SELECT * FROM orders WHERE id=?");
        mysqli_stmt_bind_param($q, 'i', $order_id); mysqli_stmt_execute($q);
        $res = mysqli_stmt_get_result($q);
        $order = mysqli_fetch_assoc($res);
        if (!$order) { answer_callback($cb['id'], 'سفارش یافت نشد', true); send_ok_and_exit(); }
        if (intval($order['telegram_id']) !== intval($cb_from)) { answer_callback($cb['id'], 'شما مجاز نیستید', true); send_ok_and_exit(); }
        mysqli_query($conn, "UPDATE orders SET status='cancelled' WHERE id=" . $order_id);
        edit_message_text($cb_msg['chat']['id'], $cb_msg['message_id'], "🚫 سفارش توسط کاربر لغو شد.");
        answer_callback($cb['id'], 'لغو شد', false);
        send_ok_and_exit();
    }

    // fallback
    answer_callback($cb['id'], 'دستور نامشخص', true);
    send_ok_and_exit();
}

/* ---------- If any admin-state persisted but no more updates, just exit ---------- */
http_response_code(200);
exit;
