<?php
// zarinpal.php - v4 endpoints (no disk logging) + normalized return shape
// Provides:
//   zarinpal_create_payment($amount, $order_id, $description = '', $metadata = null, $callback_override = null)
//   zarinpal_verify_payment($authority, $amount = null)
//
// Note: This file intentionally does NOT write API responses to disk (per request).

$config = require __DIR__ . '/config.php';

/**
 * Helper: build StartPay link from authority and sandbox flag
 */
function zarinpal_startpay_link_for_authority(?string $authority): ?string {
    global $config;
    if (!$authority) return null;
    // NOTE: live StartPay uses www.zarinpal.com (not api.zarinpal.com)
    $base = (!empty($config['zarinpal_sandbox'])) ? 'https://sandbox.zarinpal.com/pg/StartPay/' : 'https://www.zarinpal.com/pg/StartPay/';
    return $base . $authority;
}

/**
 * Create payment (normalized result)
 *
 * @param float|int $amount
 * @param int $order_id
 * @param string $description
 * @param array|null $metadata
 * @param string|null $callback_override
 * @return array {
 *   success: bool,
 *   authority: string|null,
 *   link: string|null,
 *   data: array|null,
 *   decoded: array|null,
 *   http_code: int,
 *   curl_error: string,
 *   raw: string|null,
 *   error: string|null
 * }
 */
function zarinpal_create_payment($amount, $order_id, $description = '', $metadata = null, $callback_override = null) {
    global $config;
    $amount = $amount * 10;

    $merchant = trim($config['zarinpal_merchant_id'] ?? '');
    if ($merchant === '') {
        return ['success'=>false, 'error'=>'تنظیمات درگاه ناقص است (merchant_id موجود نیست).', 'authority'=>null, 'link'=>null];
    }

    // Ensure integer amount (Zarinpal expects integer)
    $amount_int = intval(round(floatval($amount)));
    if ($amount_int <= 0) {
        return ['success'=>false, 'error'=>'مبلغ نامعتبر است.', 'authority'=>null, 'link'=>null];
    }

    $callback_base = $callback_override ? rtrim($callback_override, '/') : rtrim($config['zarinpal_callback'] ?? '', '/');
    $callback_full = $callback_base . '?order_id=' . intval($order_id) . '&amount=' . $amount_int;

    $body = [
        "merchant_id"  => $merchant,
        "amount"       => $amount_int,
        "callback_url" => $callback_full,
        "description"  => $description
    ];
    
    write_log("$merchant");
    if (!empty($metadata) && is_array($metadata)) {
        $body['metadata'] = $metadata;
    }

    $base = (!empty($config['zarinpal_sandbox'])) ? 'https://sandbox.zarinpal.com' : 'https://api.zarinpal.com';
    $url  = $base . '/pg/v4/payment/request.json';

    $json = json_encode($body, JSON_UNESCAPED_UNICODE);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'ZarinPal Rest Api v1');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json',
        'Content-Length: ' . strlen($json)
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

    $resp = curl_exec($ch);
    $curl_err = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Retry with disabled SSL verify only if SSL error detected (diagnostic) — do NOT log to disk
    if ($curl_err && (stripos($curl_err, 'SSL') !== false || stripos($curl_err, 'certificate') !== false)) {
        $ch2 = curl_init($url);
        curl_setopt($ch2, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch2, CURLOPT_POST, true);
        curl_setopt($ch2, CURLOPT_USERAGENT, 'ZarinPal Rest Api v1');
        curl_setopt($ch2, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json',
            'Content-Length: ' . strlen($json)
        ]);
        curl_setopt($ch2, CURLOPT_POSTFIELDS, $json);
        curl_setopt($ch2, CURLOPT_TIMEOUT, 20);
        curl_setopt($ch2, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch2, CURLOPT_SSL_VERIFYPEER, false);
        $resp2 = curl_exec($ch2);
        $curl_err2 = curl_error($ch2);
        $http2 = curl_getinfo($ch2, CURLINFO_HTTP_CODE);
        curl_close($ch2);

        // prefer retry's response if it returned something
        if ($resp2 !== false && $resp2 !== null && strlen(trim((string)$resp2)) > 0) {
            $resp = $resp2;
            $curl_err = $curl_err2;
            $http = $http2;
        }
    }

    $decoded = null;
    if ($resp !== false && $resp !== null && strlen(trim((string)$resp)) > 0) {
        $decoded = @json_decode($resp, true);
        if ($decoded === null) {
            // invalid JSON
        }
    }

    // Try to extract authority and link from various possible shapes
    $authority = null;
    $link = null;
    if (is_array($decoded)) {
        if (isset($decoded['data']) && is_array($decoded['data'])) {
            $d = $decoded['data'];
            $authority = $authority ?? ($d['authority'] ?? null);
            $link = $link ?? ($d['link'] ?? null);
            // older variations
            if (isset($d['code']) && intval($d['code']) !== 100) {
                // API returned non-success code
                $err_msg = $d['message'] ?? ('خطای درگاه، کد: ' . intval($d['code']));
                return [
                    'success' => false,
                    'error' => "خطا در ایجاد پرداخت: {$err_msg}",
                    'authority'=>null,
                    'link'=>null,
                    'decoded'=>$decoded,
                    'raw'=>$resp,
                    'http_code'=>intval($http),
                    'curl_error'=>$curl_err ?: ''
                ];
            }
        }
        // sometimes top-level 'data' not present but 'errors' exist
        if (isset($decoded['errors']) && is_array($decoded['errors'])) {
            $err = $decoded['errors'];
            $msg = $err['message'] ?? json_encode($err, JSON_UNESCAPED_UNICODE);
            return [
                'success' => false,
                'error' => "خطای درگاه: {$msg}",
                'authority'=>null,
                'link'=>null,
                'decoded'=>$decoded,
                'raw'=>$resp,
                'http_code'=>intval($http),
                'curl_error'=>$curl_err ?: ''
            ];
        }
    }

    // If authority present construct link
    if (!$link && $authority) {
        $link = zarinpal_startpay_link_for_authority($authority);
    }

    // If decoded indicates success but no authority, treat as error
    if (!$authority) {
        return [
            'success' => false,
            'error' => 'درخواست پرداخت پاسخ معتبری دریافت نکرد (authority پیدا نشد).',
            'authority' => null,
            'link' => null,
            'decoded' => $decoded,
            'raw' => $resp,
            'http_code' => intval($http),
            'curl_error' => $curl_err ?: ''
        ];
    }

    return [
        'success' => true,
        'authority' => $authority,
        'link' => $link,
        'data' => $decoded['data'] ?? null,
        'decoded' => $decoded,
        'raw' => $resp,
        'http_code' => intval($http),
        'curl_error' => $curl_err ?: '',
        'error' => null
    ];
}

/**
 * Verify payment (normalized result)
 *
 * @param string $authority
 * @param int|float|null $amount
 * @return array {
 *   success: bool,
 *   ref_id: string|null,
 *   data: array|null,
 *   decoded: array|null,
 *   http_code: int,
 *   curl_error: string,
 *   raw: string|null,
 *   error: string|null
 * }
 */
function zarinpal_verify_payment($authority, $amount = null) {
    global $config;
    $merchant = trim($config['zarinpal_merchant_id'] ?? '');
    if ($merchant === '') {
        return ['success'=>false, 'error'=>'تنظیمات درگاه ناقص است (merchant_id موجود نیست).'];
    }
    if (!$authority) {
        return ['success'=>false, 'error'=>'مرجع پرداخت (authority) نامعتبر است.'];
    }

    $body = [
        "merchant_id" => $merchant,
        "authority"   => $authority
    ];
    if ($amount !== null) {
        $body['amount'] = intval(round(floatval($amount)));
    }

    $base = (!empty($config['zarinpal_sandbox'])) ? 'https://sandbox.zarinpal.com' : 'https://api.zarinpal.com';
    $url  = $base . '/pg/v4/payment/verify.json';

    $json = json_encode($body, JSON_UNESCAPED_UNICODE);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'ZarinPal Rest Api v1');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json',
        'Content-Length: ' . strlen($json)
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

    $resp = curl_exec($ch);
    $curl_err = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Retry on SSL errors (no disk logging)
    if ($curl_err && (stripos($curl_err, 'SSL') !== false || stripos($curl_err, 'certificate') !== false)) {
        $ch2 = curl_init($url);
        curl_setopt($ch2, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch2, CURLOPT_POST, true);
        curl_setopt($ch2, CURLOPT_USERAGENT, 'ZarinPal Rest Api v1');
        curl_setopt($ch2, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json',
            'Content-Length: ' . strlen($json)
        ]);
        curl_setopt($ch2, CURLOPT_POSTFIELDS, $json);
        curl_setopt($ch2, CURLOPT_TIMEOUT, 20);
        curl_setopt($ch2, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch2, CURLOPT_SSL_VERIFYPEER, false);
        $resp2 = curl_exec($ch2);
        $curl_err2 = curl_error($ch2);
        $http2 = curl_getinfo($ch2, CURLINFO_HTTP_CODE);
        curl_close($ch2);

        if ($resp2 !== false && $resp2 !== null && strlen(trim((string)$resp2)) > 0) {
            $resp = $resp2;
            $curl_err = $curl_err2;
            $http = $http2;
        }
    }

    $decoded = null;
    if ($resp !== false && $resp !== null && strlen(trim((string)$resp)) > 0) {
        $decoded = @json_decode($resp, true);
    }

    // parse outcome
    $data = $decoded['data'] ?? null;
    $code = null;
    if (is_array($data) && isset($data['code'])) $code = intval($data['code']);

    if ($code === 100 || (is_array($decoded) && isset($decoded['data']['code']) && intval($decoded['data']['code']) === 100)) {
        $ref_id = $data['ref_id'] ?? $decoded['data']['ref_id'] ?? null;
        return [
            'success' => true,
            'ref_id' => $ref_id,
            'data' => $data,
            'decoded' => $decoded,
            'raw' => $resp,
            'http_code' => intval($http),
            'curl_error' => $curl_err ?: '',
            'error' => null
        ];
    }

    // if errors present
    if (is_array($decoded) && isset($decoded['errors'])) {
        $err = $decoded['errors'];
        $msg = $err['message'] ?? json_encode($err, JSON_UNESCAPED_UNICODE);
        return [
            'success' => false,
            'ref_id' => null,
            'data' => $data,
            'decoded' => $decoded,
            'raw' => $resp,
            'http_code' => intval($http),
            'curl_error' => $curl_err ?: '',
            'error' => "خطای درگاه: {$msg}"
        ];
    }

    // fallback error
    return [
        'success' => false,
        'ref_id' => null,
        'data' => $data,
        'decoded' => $decoded,
        'raw' => $resp,
        'http_code' => intval($http),
        'curl_error' => $curl_err ?: '',
        'error' => 'تایید پرداخت موفقیت آمیز نبود یا پاسخ نامشخص بود.'
    ];
}