<?php
// payment_callback.php
// Callback handler for Zarinpal (updated — handles traffic in BYTES and tries both client/object and {"clients":[...]} wrapper)
// - Uses normalized zarinpal.php (zarinpal_verify_payment should return ['success'=>bool, ...])
// - No disk logging
// - Never reveal panel credentials (only server_name shown to admins)
// - Persian messages & emojis
// - Prepared statements for DB access
// - Attempts to update panel client in two shapes: client object or {"clients":[client]}

require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/zarinpal.php';
$config = require __DIR__ . '/config.php';
$conn = db_connect();

// Helper: fetch single assoc using prepared stmt
function fetch_one_assoc($conn, $sql, $types = '', $params = []) {
    $stmt = mysqli_prepare($conn, $sql);
    if (!$stmt) return null;
    if ($types !== '') {
        mysqli_stmt_bind_param($stmt, $types, ...$params);
    }
    mysqli_stmt_execute($stmt);
    $res = mysqli_stmt_get_result($stmt);
    if (!$res) { mysqli_stmt_close($stmt); return null; }
    $row = mysqli_fetch_assoc($res);
    mysqli_stmt_close($stmt);
    return $row ? $row : null;
}

// Read params (Zarinpal may call with GET or POST)
$amount_in = $_REQUEST['amount'] ?? null;
$order_id  = isset($_REQUEST['order_id']) ? intval($_REQUEST['order_id']) : 0;
$authority = $_REQUEST['Authority'] ?? ($_REQUEST['authority'] ?? null);

// Normalize amount (gateway might send string)
$amount = null;
if ($amount_in !== null && $amount_in !== '') {
    // Use intval(round(floatval())) to normalize decimal or string amounts
    $amount = intval(round(floatval($amount_in)));
}

// Basic validation
if (!$order_id || !$authority) {
    http_response_code(400);
    echo "پارامترهای لازم ارسال نشده‌اند.";
    exit;
}

// If amount missing, try payments table by provider_ref
if (!$amount || $amount <= 0) {
    $prow = fetch_one_assoc($conn, "SELECT amount FROM payments WHERE provider_ref = ? LIMIT 1", "s", [$authority]);
    if ($prow && isset($prow['amount'])) {
        $amount = intval(round(floatval($prow['amount'])));
    }
}

// Call zarinpal verify (uses normalized return shape)
$ver = null;
try {
    if (function_exists('zarinpal_verify_payment')) {
        $ver = ($amount && $amount > 0) ? zarinpal_verify_payment($authority, $amount) : zarinpal_verify_payment($authority, null);
    } else {
        $ver = ['success'=>false, 'error'=>'توابع درگاه پیکربندی نشده‌اند.'];
    }
} catch (Throwable $e) {
    $ver = ['success'=>false, 'error' => 'خطا در تماس با درگاه: ' . $e->getMessage()];
}

// Determine success, support legacy shape
$verified = (isset($ver['success']) && $ver['success'] === true);
if (!$verified && isset($ver['decoded']['data']['code']) && intval($ver['decoded']['data']['code']) === 100) {
    $verified = true;
}

if ($verified) {
    // mark payment verified (if exists)
    $up1 = mysqli_prepare($conn, "UPDATE payments SET status='verified' WHERE provider_ref = ?");
    if ($up1) {
        mysqli_stmt_bind_param($up1, 's', $authority);
        mysqli_stmt_execute($up1);
        mysqli_stmt_close($up1);
    }

    // mark order paid
    $up2 = mysqli_prepare($conn, "UPDATE orders SET status='paid' WHERE id = ?");
    if ($up2) {
        mysqli_stmt_bind_param($up2, 'i', $order_id);
        mysqli_stmt_execute($up2);
        mysqli_stmt_close($up2);
    }

    // fetch order
    $order = fetch_one_assoc($conn, "SELECT * FROM orders WHERE id = ? LIMIT 1", "i", [$order_id]);
    if (!$order) {
        http_response_code(404);
        echo "سفارش پیدا نشد.";
        exit;
    }

    // Determine additions: add_hours (hours) and add_bytes (bytes)
    $add_hours = 0;
    $add_bytes = 0;

    // Preferred: plan_id referencing services_plans
    if (!empty($order['plan_id'])) {
        $plan = fetch_one_assoc($conn, "SELECT * FROM services_plans WHERE id = ? LIMIT 1", "i", [intval($order['plan_id'])]);
        if ($plan) {
            // services_plans.duration_days (days) -> hours
            $duration_days = intval($plan['duration_days'] ?? 0);
            $add_hours = $duration_days * 24;
            // amount is stored in bytes (per your DB example)
            $add_bytes = intval($plan['amount'] ?? 0);
        }
    }

    // Fallback: legacy duration_id / data_id
    if (($add_hours === 0 && !empty($order['duration_id'])) || ($add_bytes === 0 && !empty($order['data_id']))) {
        if (!empty($order['duration_id'])) {
            $drow = fetch_one_assoc($conn, "SELECT * FROM services_duration WHERE id = ? LIMIT 1", "i", [intval($order['duration_id'])]);
            if ($drow) {
                // assume legacy duration amount stored in hours
                $add_hours = intval($drow['amount'] ?? 0);
            }
        }
        if (!empty($order['data_id'])) {
            $srow = fetch_one_assoc($conn, "SELECT * FROM services_data WHERE id = ? LIMIT 1", "i", [intval($order['data_id'])]);
            if ($srow) {
                // assume legacy data amount stored in bytes
                $add_bytes = intval($srow['amount'] ?? 0);
            }
        }
    }

    // Defensive: accept explicit fields on order
    if ($add_hours === 0 && !empty($order['add_hours'])) {
        $add_hours = intval($order['add_hours']);
    }
    if ($add_bytes === 0 && !empty($order['add_bytes'])) {
        $add_bytes = intval($order['add_bytes']);
    }

    // If panel assignment exists, try to apply automatically
    if (!empty($order['panel_id']) && !empty($order['panel_account_id'])) {
        $panel = fetch_one_assoc($conn, "SELECT * FROM services_panels WHERE id = ? LIMIT 1", "i", [intval($order['panel_id'])]);
        if ($panel) {
            $matched = null;
            $inbounds = null;
            if (function_exists('panel_get_inbounds_list')) {
                try {
                    $inbounds = panel_get_inbounds_list($panel);
                } catch (Throwable $e) {
                    $inbounds = null;
                }
            }

            if (is_array($inbounds)) {
                foreach ($inbounds as $inb) {
                    $settings = $inb['settings'] ?? null;
                    if (!is_array($settings)) {
                        $decoded = @json_decode($settings, true);
                        if ($decoded !== null) $settings = $decoded;
                    }
                    if (is_array($settings) && isset($settings['clients']) && is_array($settings['clients'])) {
                        foreach ($settings['clients'] as $client) {
                            $cid = $client['id'] ?? $client['uuid'] ?? $client['subId'] ?? null;
                            if ($cid && ((string)$cid === (string)$order['panel_account_id'])) {
                                $matched = ['inbound'=>$inb, 'client'=>$client];
                                break 2;
                            }
                        }
                    }
                    if (!empty($inb['clientStats']) && is_array($inb['clientStats'])) {
                        foreach ($inb['clientStats'] as $clientStat) {
                            $cid = $clientStat['id'] ?? $clientStat['uuid'] ?? $clientStat['subId'] ?? null;
                            if ($cid && ((string)$cid === (string)$order['panel_account_id'])) {
                                $matched = ['inbound'=>$inb, 'client'=>$clientStat];
                                break 2;
                            }
                        }
                    }
                }
            }

            if ($matched) {
                $inbound = $matched['inbound'];
                $client = $matched['client'];

                // expiryTime is in milliseconds in panel examples
                $existing_expiry = intval($client['expiryTime'] ?? 0);
                // add hours (in ms) to either now (if expired) or to the existing expiry (if still active)
                $add_ms = intval($add_hours) * 3600 * 1000;
                $now_ms = (int) (microtime(true) * 1000);
                
                if ($existing_expiry <= $now_ms) {
                    // already expired -> start from now
                    $new_expiry = $now_ms + $add_ms;
                } else {
                    // still active -> extend from current expiry
                    $new_expiry = $existing_expiry + $add_ms;
                }

                // IMPORTANT: panel stores totalGB as BYTES (per your sample)
                $existing_total_bytes = intval($client['totalGB'] ?? 0);
                $add_bytes_int = intval($add_bytes);
                $new_total_bytes = $existing_total_bytes + $add_bytes_int;

                // Build updated client object matching your example structure
                $updated_client = $client; // start from existing client to preserve unknown fields
                $updated_client['expiryTime'] = $new_expiry;            // milliseconds
                $updated_client['totalGB']    = $add_bytes_int;       // bytes (exactly as panel expects)
                $updated_client['enable']     = true;
                $updated_client['flow']       = $client['flow'] ?? '';
                // limitIp present in your example — if panel requires integer
                if (isset($client['limitIp'])) {
                    $updated_client['limitIp'] = intval($client['limitIp']);
                } else {
                    $updated_client['limitIp'] = $client['limitIp'] ?? 1;
                }
                $updated_client['reset']      = 0;
                $updated_client['email']      = $client['email'] ?? ($order['reference'] ?? '');
                $updated_client['tgId']       = $client['tgId'] ?? '';
                // keep subId/id unchanged (used to identify)
                $client_uuid = $client['id'] ?? $client['uuid'] ?? null;
                $inbound_id = $inbound['id'] ?? 0;

                $panel_name = $panel['server_name'] ?? ('پنل #' . intval($panel['id']));

                // Try updating panel: attempt with client object first, then wrapper {"clients":[...]} if fail
                $upd_result = null;
                if ($client_uuid && $inbound_id && function_exists('panel_update_client_by_uuid')) {
                    try {
                        $upd_result = panel_update_client_by_uuid($panel, $client_uuid, $inbound_id, $updated_client);
                    } catch (Throwable $e) {
                        $upd_result = null;
                    }

                    // If falsey result, try wrapper shape (some panels expect full clients array)
                    if (!$upd_result) {
                        try {
                            $upd_result = panel_update_client_by_uuid($panel, $client_uuid, $inbound_id, ['clients' => [$updated_client]]);
                        } catch (Throwable $e) {
                            $upd_result = null;
                        }
                    }

                    // Decide success based on truthiness of $upd_result
                    if ($upd_result) {
                        // success: notify user + admins
                        $human_add = [];
                        if ($add_hours > 0) $human_add[] = "+{$add_hours} ساعت";
                        if ($add_bytes > 0) $human_add[] = "+" . format_bytes($add_bytes);
                        $human_add_text = $human_add ? implode('، ', $human_add) : "تغییر اعمال شد";

                        if (function_exists('telegram_send_message')) telegram_send_message($order['telegram_id'], "🎉 پرداخت شما دریافت و اعمال شد");
                        $msg_admin = "✅ سفارش {$order_id} پرداخت و روی پنل {$panel_name} اعمال شد.\nکاربر: {$order['telegram_id']}";
                        if (function_exists('telegram_send_to_admins')) telegram_send_to_admins($msg_admin);
                    } else {
                        // failed to apply programmatically
                        $msg_admin = "⚠️ پرداخت تایید شد ولی اعمال خودکار روی پنل ناموفق بود.\nسفارش: {$order_id}\nپنل: {$panel_name}\nکاربر: {$order['telegram_id']}";
                        if (function_exists('telegram_send_to_admins')) telegram_send_to_admins($msg_admin);
                        if (function_exists('telegram_send_message')) telegram_send_message($order['telegram_id'], "✅ پرداخت شما دریافت شد ولی اعمال خودکار روی پنل انجام نشد؛ مدیران مطلع شدند و بررسی می‌کنند.");
                    }
                } else {
                    // missing identifiers or no update function
                    $admin_msg = "ℹ️ پرداخت تایید شد اما شناسهٔ کاربر در پنل پیدا نشد یا تابع به‌روزرسانی پنل موجود نیست.\nسفارش: {$order_id}\nپنل: {$panel_name}\nکاربر: {$order['telegram_id']}";
                    if (function_exists('telegram_send_to_admins')) telegram_send_to_admins($admin_msg);
                    if (function_exists('telegram_send_message')) telegram_send_message($order['telegram_id'], "✅ پرداخت شما ثبت شد. اعمال سرویس به‌صورت دستی انجام خواهد شد.");
                }
            } else {
                // user not found in inbounds
                $panel_name = $panel['server_name'] ?? ('پنل #' . intval($panel['id']));
                $admin_msg = "ℹ️ پرداخت تایید شد اما کاربر در inbounds پنل پیدا نشد.\nسفارش: {$order_id}\nپنل: {$panel_name}\nکاربر: {$order['telegram_id']}";
                if (function_exists('telegram_send_to_admins')) telegram_send_to_admins($admin_msg);
                if (function_exists('telegram_send_message')) telegram_send_message($order['telegram_id'], "✅ پرداخت دریافت شد. مدیران برای اعمال دستی مطلع شدند.");
            }
        } else {
            // panel row missing
            $admin_msg = "⚠️ پرداخت تایید شد اما رکورد پنل با id {$order['panel_id']} در دیتابیس یافت نشد. سفارش: {$order_id}";
            if (function_exists('telegram_send_to_admins')) telegram_send_to_admins($admin_msg);
            if (function_exists('telegram_send_message')) telegram_send_message($order['telegram_id'], "✅ پرداخت دریافت شد اما پنل مرتبط یافت نشد؛ مدیران بررسی می‌کنند.");
        }
    } else {
        // order has no panel assigned
        $admin_msg = "✅ پرداخت تایید شد اما سفارش {$order_id} بدون پنل بوده است. کاربر: {$order['telegram_id']}";
        if (function_exists('telegram_send_to_admins')) telegram_send_to_admins($admin_msg);
        if (function_exists('telegram_send_message')) telegram_send_message($order['telegram_id'], "✅ پرداخت شما ثبت شد. مدیران برای فعال‌سازی سرویس مطلع شدند.");
    }

    // Minimal response for gateway
    echo "OK";
    exit;
}

// Verification failed
// mark payment failed (if payment row exists)
$stmt_fail = mysqli_prepare($conn, "UPDATE payments SET status='failed' WHERE provider_ref = ?");
if ($stmt_fail) {
    mysqli_stmt_bind_param($stmt_fail, 's', $authority);
    mysqli_stmt_execute($stmt_fail);
    mysqli_stmt_close($stmt_fail);
}

// Build error text for admins
$err_text = '';
if (!empty($ver['error'])) {
    $err_text = $ver['error'];
} elseif (!empty($ver['decoded']) && is_array($ver['decoded'])) {
    $err_text = 'پاسخ درگاه نامشخص';
} else {
    $err_text = 'تأیید پرداخت موفقیت‌آمیز نبود.';
}

$admin_msg = "❌ بررسی پرداخت ناموفق بود.\nسفارش: {$order_id}\nمرجع: {$authority}\nپیغام: {$err_text}";
if (function_exists('telegram_send_to_admins')) telegram_send_to_admins($admin_msg);

// Inform user politely (if order owner known)
$ordrow = fetch_one_assoc($conn, "SELECT telegram_id FROM orders WHERE id = ? LIMIT 1", "i", [$order_id]);
if ($ordrow && !empty($ordrow['telegram_id'])) {
    if (function_exists('telegram_send_message')) telegram_send_message($ordrow['telegram_id'], "❗ پرداخت شما تایید نشد. اگر وجه از حساب کسر شده، لطفاً صبر کنید یا با پشتیبانی تماس بگیرید.");
}

echo "FAILED";
exit;
