<?php
// functions.php
// Procedural helpers, panel login + inbounds list + updateClient helpers
require_once __DIR__ . '/db.php';
$config = require __DIR__ . '/config.php';

error_reporting(E_ALL); // Report all errors
ini_set('display_errors', 0); // Don’t show to user
ini_set('log_errors', 1); // Enable logging
ini_set('error_log', __DIR__ . '/error_log.txt'); // Log file path

/* ----------------------
   Basic helpers
   ---------------------- */
function db_connect() {
    global $config;
    static $conn = null;
    if ($conn !== null) return $conn;
    $conn = mysqli_connect($config['db_host'], $config['db_user'], $config['db_pass'], $config['db_name']);
    if (!$conn) {
        error_log("DB connect error: " . mysqli_connect_error());
        die("Database connection error");
    }
    mysqli_set_charset($conn, 'utf8mb4');
    return $conn;
}

function telegram_send_message($chat_id, $text, $reply_markup = null) {
    global $config;
    $token = $config['telegram_token'];
    $url = "https://api.telegram.org/bot{$token}/sendMessage";
    $data = ['chat_id' => $chat_id, 'text' => $text, 'parse_mode' => 'HTML'];
    if ($reply_markup) $data['reply_markup'] = json_encode($reply_markup);
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_exec($ch);
    curl_close($ch);
}

/* Add user to user_details table if not present */
function add_user_if_not_exists($telegram_id) {
    $conn = db_connect();
    $t = intval($telegram_id);
    $stmt = mysqli_prepare($conn, "SELECT id FROM user_details WHERE related_telegram_id = ?");
    mysqli_stmt_bind_param($stmt, 'i', $t);
    mysqli_stmt_execute($stmt);
    $res = mysqli_stmt_get_result($stmt);
    if ($row = mysqli_fetch_assoc($res)) return intval($row['id']);
    $ins = mysqli_prepare($conn, "INSERT INTO user_details (related_telegram_id, user_status, credit) VALUES (?, 'customer', 0.00)");
    mysqli_stmt_bind_param($ins, 'i', $t);
    mysqli_stmt_execute($ins);
    return mysqli_insert_id($conn);
}

/* Get admin Telegram IDs from user_details (user_status = 'admin') */
function get_admin_telegram_ids() {
    $conn = db_connect();
    $ids = [];
    $res = mysqli_query($conn, "SELECT related_telegram_id FROM user_details WHERE user_status='admin' AND related_telegram_id IS NOT NULL");
    if ($res) {
        while ($r = mysqli_fetch_assoc($res)) {
            if (!empty($r['related_telegram_id'])) $ids[] = intval($r['related_telegram_id']);
        }
    }
    return $ids;
}

function telegram_send_to_admins($text) {
    $admins = get_admin_telegram_ids();
    foreach ($admins as $id) telegram_send_message($id, $text);
}

/* ----------------------
   Identity extractor
   ---------------------- */
function extract_identity_from_link($text) {
    $text = trim($text);
    // vmess:// (base64 JSON)
    if (stripos($text, 'vmess://') === 0) {
        $b = substr($text, 8);
        $json = @base64_decode($b);
        if ($json !== false) {
            $obj = @json_decode($json, true);
            $id = $obj['id'] ?? $obj['uuid'] ?? null;
            return ['type'=>'vmess', 'id'=>$id, 'raw'=>$text, 'parsed'=>$obj];
        }
        return ['type'=>'vmess','id'=>null,'raw'=>$text];
    }
    // vless://uuid@...
    if (stripos($text, 'vless://') === 0) {
        if (preg_match('/^vless:\/\/([^@\/\?]+)/i', $text, $m)) {
            return ['type'=>'vless','id'=>$m[1],'raw'=>$text];
        }
        return ['type'=>'vless','id'=>null,'raw'=>$text];
    }
    // trojan://password@...
    if (stripos($text, 'trojan://') === 0) {
        if (preg_match('/^trojan:\/\/([^@\/\?]+)/i', $text, $m)) {
            return ['type'=>'trojan','id'=>$m[1],'raw'=>$text];
        }
        return ['type'=>'trojan','id'=>null,'raw'=>$text];
    }
    // ss://
    if (stripos($text, 'ss://') === 0) {
        $rest = substr($text, 5);
        $rest = preg_replace('/#.*$/','',$rest);
        if (strpos($rest, '@') === false) {
            $dec = @base64_decode($rest);
            if ($dec !== false && strpos($dec, ':') !== false) {
                if (preg_match('/^([^:]+):([^@]+)@?(.*)$/', $dec, $mm)) {
                    return ['type'=>'ss','id'=>$mm[2],'method'=>$mm[1],'raw'=>$text];
                }
            }
        } else {
            if (preg_match('/^([^:]+):([^@]+)@/', $rest, $mm)) {
                return ['type'=>'ss','id'=>$mm[2],'method'=>$mm[1],'raw'=>$text];
            }
            return ['type'=>'ss','id'=>null,'raw'=>$text];
        }
        return ['type'=>'ss','id'=>null,'raw'=>$text];
    }
    return ['type'=>'unknown','id'=>null,'raw'=>$text];
}

/* ----------------------
   Panel debug logging helpers
   ---------------------- */
function ensure_logs_dir() {
    $dir = __DIR__ . '/logs';
    if (!is_dir($dir)) @mkdir($dir, 0755, true);
    return $dir;
}

function panel_debug_log($panel_id, $msg) {
    $dir = ensure_logs_dir();
    $file = "{$dir}/panel_{$panel_id}.log";
    $line = '[' . gmdate('Y-m-d H:i:s') . ' UTC] ' . $msg . PHP_EOL;
    @file_put_contents($file, $line, FILE_APPEND | LOCK_EX);
}

/* ----------------------
   Panel login & API helpers (with verbose logging)
   ---------------------- */

/* Return path to cookie file for panel */
function panel_cookiefile($panel_id) {
    $tmp = sys_get_temp_dir();
    return $tmp . DIRECTORY_SEPARATOR . "panel_cookie_{$panel_id}.txt";
}

/* Login to panel and store cookies in cookiefile (returns true on success) */
function panel_login($panel) {
    $dir = ensure_logs_dir();
    $cookiefile = panel_cookiefile($panel['id']);
    $urlBase = rtrim($panel['url'], '/');

    $loginUrlCandidates = [
        $urlBase . '/login',
        $urlBase . '/login/',
    ];

    $payload = ['username' => $panel['username'], 'password' => $panel['password']];

    foreach ($loginUrlCandidates as $loginUrl) {
        // Try form-encoded
        $ch = curl_init($loginUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($payload));
        curl_setopt($ch, CURLOPT_COOKIEJAR, $cookiefile);
        curl_setopt($ch, CURLOPT_COOKIEFILE, $cookiefile);
        curl_setopt($ch, CURLOPT_TIMEOUT, 12);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Accept: application/json']);
        $out = curl_exec($ch);
        $err = curl_error($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        panel_debug_log($panel['id'], "LOGIN(form) attempt to {$loginUrl} HTTP={$http} curl_err=" . ($err ?: 'none') . " resp_preview=" . substr($out ?? '', 0, 800));
        @file_put_contents($dir . "/panel_{$panel['id']}_login_resp.txt", ($out ?? ''));

        if ($err) {
            // curl error - try next
        } else {
            if ($http >= 200 && $http < 400 && file_exists($cookiefile) && filesize($cookiefile) > 10) {
                panel_debug_log($panel['id'], "LOGIN(form) success: cookie created ({$cookiefile})");
                return true;
            }
        }

        // Try JSON body login
        $ch = curl_init($loginUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        $jsonBody = json_encode($payload);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonBody);
        curl_setopt($ch, CURLOPT_COOKIEJAR, $cookiefile);
        curl_setopt($ch, CURLOPT_COOKIEFILE, $cookiefile);
        curl_setopt($ch, CURLOPT_TIMEOUT, 12);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Accept: application/json', 'Content-Type: application/json']);
        $out2 = curl_exec($ch);
        $err2 = curl_error($ch);
        $http2 = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        panel_debug_log($panel['id'], "LOGIN(json) attempt to {$loginUrl} HTTP={$http2} curl_err=" . ($err2 ?: 'none') . " resp_preview=" . substr($out2 ?? '', 0, 800));
        @file_put_contents($dir . "/panel_{$panel['id']}_login_resp.txt", ($out2 ?? ''), FILE_APPEND | LOCK_EX);

        if (!$err2 && $http2 >= 200 && $http2 < 400 && file_exists($cookiefile) && filesize($cookiefile) > 10) {
            panel_debug_log($panel['id'], "LOGIN(json) success: cookie created ({$cookiefile})");
            return true;
        }
    }

    panel_debug_log($panel['id'], "LOGIN failed for all endpoints; cookiefile exists? " . (file_exists($cookiefile) ? 'yes' : 'no'));
    return false;
}

/* GET inbounds list using panel session (login if needed) */
function panel_get_inbounds_list($panel) {
    $dir = ensure_logs_dir();
    $cookiefile = panel_cookiefile($panel['id']);
    // ensure logged in
    if (!file_exists($cookiefile) || filesize($cookiefile) < 10) {
        panel_debug_log($panel['id'], "Cookie missing/small; attempting login");
        $ok = panel_login($panel);
        panel_debug_log($panel['id'], "Login result: " . ($ok ? 'true' : 'false'));
        if (!$ok) {
            panel_debug_log($panel['id'], "Aborting get_inbounds_list due to login failure");
            return null;
        }
    }

    $urlBase = rtrim($panel['url'], '/');
    $candidates = [
        $urlBase . '/panel/api/inbounds/list',
        $urlBase . '/api/inbounds/list',
        $urlBase . '/panel/api/inbounds/list/'
    ];
    foreach ($candidates as $url) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, $cookiefile);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Accept: application/json']);
        $out = curl_exec($ch);
        $err = curl_error($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        panel_debug_log($panel['id'], "GET inbounds attempt {$url} HTTP={$http} curl_err=" . ($err ?: 'none') . " resp_len=" . strlen($out));
        @file_put_contents($dir . "/panel_{$panel['id']}.json", $out);

        if ($err) {
            panel_debug_log($panel['id'], "cURL error fetching inbounds: {$err}");
            continue;
        }
        if (!$out || $http < 200 || $http >= 400) {
            panel_debug_log($panel['id'], "Non-success HTTP code {$http} or empty body from {$url}");
            continue;
        }
        $decoded = @json_decode($out, true);
        if ($decoded === null) {
            panel_debug_log($panel['id'], "JSON decode failed for inbounds response; preview: " . substr($out,0,800));
            continue;
        }
        $list = $decoded['obj'] ?? $decoded;
        if (!is_array($list)) {
            panel_debug_log($panel['id'], "Inbounds response decoded but 'obj' not array and root not array; root type: " . gettype($list));
            continue;
        }
        panel_debug_log($panel['id'], "Got inbounds list with count=" . count($list) . " from {$url}");
        return $list;
    }
    panel_debug_log($panel['id'], "All inbounds list endpoints failed or returned no usable data");
    return null;
}

/* Find account in inbound list using identity (identity = extract_identity_from_link result or plain string)
   returns array with panel, inbound, client and which field matched */
function panel_find_account_via_inbounds($panel, $identity) {
    $needle = is_array($identity) ? ($identity['id'] ?? null) : (string)$identity;
    $needle_raw = is_array($identity) ? ($identity['raw'] ?? null) : $needle;
    if (!$needle && !$needle_raw) return null;

    $list = panel_get_inbounds_list($panel);
    if (!is_array($list)) return null;

    foreach ($list as $inbound) {
        // check clientStats if present
        if (!empty($inbound['clientStats']) && is_array($inbound['clientStats'])) {
            foreach ($inbound['clientStats'] as $client) {
                $fields = [];
                if (isset($client['uuid'])) $fields[] = $client['uuid'];
                if (isset($client['id'])) $fields[] = $client['id'];
                if (isset($client['subId'])) $fields[] = $client['subId'];
                if (isset($client['email'])) $fields[] = $client['email'];
                foreach ($fields as $f) {
                    if (!$f) continue;
                    if ($needle && strcasecmp((string)$f, (string)$needle) === 0) {
                        return ['panel'=>$panel, 'inbound'=>$inbound, 'client'=>$client, 'matched_field'=>$f, 'matched_by'=>'clientStats'];
                    }
                    if ($needle_raw && stripos((string)$f, (string)$needle_raw) !== false) {
                        return ['panel'=>$panel, 'inbound'=>$inbound, 'client'=>$client, 'matched_field'=>$f, 'matched_by'=>'clientStats_partial'];
                    }
                }
            }
        }

        // check settings.clients (sometimes JSON string)
        if (!empty($inbound['settings'])) {
            $settings = $inbound['settings'];
            if (!is_array($settings)) {
                $decoded = @json_decode($settings, true);
                if ($decoded !== null) $settings = $decoded;
            }
            if (is_array($settings) && isset($settings['clients']) && is_array($settings['clients'])) {
                foreach ($settings['clients'] as $client) {
                    $fields = [];
                    if (isset($client['id'])) $fields[] = $client['id'];
                    if (isset($client['uuid'])) $fields[] = $client['uuid'];
                    if (isset($client['subId'])) $fields[] = $client['subId'];
                    if (isset($client['email'])) $fields[] = $client['email'];
                    if (isset($client['comment'])) $fields[] = $client['comment'];
                    foreach ($fields as $f) {
                        if (!$f) continue;
                        if ($needle && strcasecmp((string)$f, (string)$needle) === 0) {
                            return ['panel'=>$panel, 'inbound'=>$inbound, 'client'=>$client, 'matched_field'=>$f, 'matched_by'=>'settings_clients'];
                        }
                        if ($needle_raw && stripos((string)$f, (string)$needle_raw) !== false) {
                            return ['panel'=>$panel, 'inbound'=>$inbound, 'client'=>$client, 'matched_field'=>$f, 'matched_by'=>'settings_clients_partial'];
                        }
                    }
                }
            }
        }

        // fallback: check inbound tag/port/protocol
        $fallback_fields = [];
        if (isset($inbound['tag'])) $fallback_fields[] = $inbound['tag'];
        if (isset($inbound['port'])) $fallback_fields[] = (string)$inbound['port'];
        if (isset($inbound['protocol'])) $fallback_fields[] = $inbound['protocol'];
        foreach ($fallback_fields as $f) {
            if (!$f) continue;
            if ($needle_raw && stripos((string)$f, (string)$needle_raw) !== false) {
                return ['panel'=>$panel, 'inbound'=>$inbound, 'client'=>null, 'matched_field'=>$f, 'matched_by'=>'inbound_partial'];
            }
        }
    }
    return null;
}

/**
 * Fetch client traffics from panel API: /panel/api/inbounds/getClientTraffics/{email}
 * Ensures login cookie exists (calls panel_login) and retries once on failure.
 *
 * Returns decoded array on success (the JSON-decoded response), or null on failure.
 */
function panel_get_client_traffics($panel, $email) {
    // prepare log dir (use ensure_logs_dir() if available)
    if (function_exists('ensure_logs_dir')) {
        $dir = ensure_logs_dir();
    } else {
        $dir = __DIR__ . '/logs';
        if (!is_dir($dir)) @mkdir($dir, 0755, true);
    }

    $panel_id = intval($panel['id'] ?? 0);
    $cookiefile = panel_cookiefile($panel_id);
    $urlBase = rtrim($panel['url'] ?? '', '/');

    if (empty($urlBase) || empty($email)) {
        if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_get_client_traffics: missing urlBase or email");
        return null;
    }

    // candidate endpoints (primary + fallback)
    $candidates = [
        $urlBase . "/panel/api/inbounds/getClientTraffics/" . rawurlencode($email),
        $urlBase . "/api/inbounds/getClientTraffics/" . rawurlencode($email),
        $urlBase . "/panel/api/inbounds/getClientTraffics/" . rawurlencode($email) . "/",
    ];

    // ensure cookie + login available, try to create cookie if missing/small
    $need_login = (!file_exists($cookiefile) || filesize($cookiefile) < 10);
    if ($need_login) {
        if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_get_client_traffics: cookie missing or too small, attempting panel_login()");
        $ok = false;
        try {
            $ok = panel_login($panel);
            if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_login initial attempt returned: " . ($ok ? 'true' : 'false'));
        } catch (Throwable $e) {
            if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_login threw: " . $e->getMessage());
            $ok = false;
        }
        if (!$ok) {
            // remove stale cookie and retry once
            if (file_exists($cookiefile)) @unlink($cookiefile);
            try {
                $ok = panel_login($panel);
                if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_login second attempt returned: " . ($ok ? 'true' : 'false'));
            } catch (Throwable $e) {
                if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_login second attempt threw: " . $e->getMessage());
                $ok = false;
            }
        }
    } else {
        if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_get_client_traffics: cookie exists (size=" . filesize($cookiefile) . ")");
    }

    // if still no cookie, log and bail out
    if (!file_exists($cookiefile) || filesize($cookiefile) < 10) {
        if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_get_client_traffics: no valid cookie after login attempts - skipping traffic request");
        return null;
    }

    // request function with SSL-retry logic
    $do_request = function($endpoint) use ($cookiefile, $panel_id, $dir) {
        $ch = curl_init($endpoint);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_COOKIEFILE     => $cookiefile,
            CURLOPT_COOKIEJAR      => $cookiefile,
            CURLOPT_ENCODING       => '',
            CURLOPT_HTTPHEADER     => [
                'Accept: application/json',
                'Connection: close',
                'Expect:'
            ],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_FRESH_CONNECT  => true,
            CURLOPT_FORBID_REUSE   => true,
        ]);
        $resp = curl_exec($ch);
        $err  = curl_error($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "Traffic call to {$endpoint} HTTP={$http} curl_err=" . ($err ?: 'none') . " bytes=" . strlen($resp ?? ''));
        // save preview file for debugging
        @file_put_contents($dir . "/panel_{$panel_id}_traffics_preview.txt", "[" . gmdate('Y-m-d H:i:s') . "] endpoint={$endpoint} HTTP={$http} curl_err=" . ($err ?: 'none') . PHP_EOL . substr($resp ?? '', 0, 4000) . PHP_EOL . str_repeat('-',80).PHP_EOL, FILE_APPEND | LOCK_EX);

        // If SSL error, retry once with VERIFYPEER=false (diagnostic)
        if ($err && (stripos($err, 'SSL') !== false || stripos($err, 'certificate') !== false)) {
            if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "Traffic call SSL error detected, retrying with SSL verify disabled (diagnostic)");
            $ch2 = curl_init($endpoint);
            curl_setopt_array($ch2, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_COOKIEFILE     => $cookiefile,
                CURLOPT_COOKIEJAR      => $cookiefile,
                CURLOPT_ENCODING       => '',
                CURLOPT_HTTPHEADER     => ['Accept: application/json','Connection: close','Expect:'],
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_TIMEOUT        => 30,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_FRESH_CONNECT  => true,
                CURLOPT_FORBID_REUSE   => true,
            ]);
            $resp2 = curl_exec($ch2);
            $err2  = curl_error($ch2);
            $http2 = curl_getinfo($ch2, CURLINFO_HTTP_CODE);
            curl_close($ch2);
            if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "Traffic retry HTTP={$http2} curl_err=" . ($err2 ?: 'none') . " bytes=" . strlen($resp2 ?? ''));
            @file_put_contents($dir . "/panel_{$panel_id}_traffics_preview.txt", "[" . gmdate('Y-m-d H:i:s') . "] RETRY endpoint={$endpoint} HTTP={$http2} curl_err=" . ($err2 ?: 'none') . PHP_EOL . substr($resp2 ?? '', 0, 4000) . PHP_EOL . str_repeat('-',80).PHP_EOL, FILE_APPEND | LOCK_EX);
            return [$resp2, $err2, $http2];
        }

        return [$resp, $err, $http];
    };

    // try each candidate endpoint
    foreach ($candidates as $endpoint) {
        list($resp, $err, $http) = $do_request($endpoint);

        // if HTTP not success or curl err, attempt to re-login once and retry this endpoint
        if ($err || $http < 200 || $http >= 400 || empty($resp)) {
            if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "Traffic request to {$endpoint} failed (HTTP={$http}, err=" . ($err ?: 'none') . "). Attempting re-login and retry for this endpoint.");
            try {
                panel_login($panel);
            } catch (Throwable $e) {
                if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_login during traffic-retry threw: " . $e->getMessage());
            }
            list($resp, $err, $http) = $do_request($endpoint);
        }

        // if success, try decode and return
        if (!$err && $http >= 200 && $resp) {
            $decoded = @json_decode($resp, true);
            if ($decoded !== null) {
                // save the full raw JSON for later inspection
                @file_put_contents($dir . "/panel_{$panel_id}_traffics.json", $resp);
                if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "Traffic JSON decoded successfully from {$endpoint}");
                return $decoded;
            } else {
                if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "Traffic JSON decode failed for {$endpoint}; preview saved");
                // continue to next candidate
            }
        } else {
            if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "Traffic request to {$endpoint} final failure HTTP={$http} err=" . ($err ?: 'none'));
            // continue to next candidate
        }
    }

    // all candidates failed
    if (function_exists('panel_debug_log')) panel_debug_log($panel_id, "panel_get_client_traffics: all endpoints failed or returned unusable data");
    return null;
}


/* Update a client by uuid using POST /panel/api/inbounds/updateClient/{uuid}
   $panel: panel row
   $client_uuid: client's unique id string (client['id'] or client['uuid'])
   $inbound_id: inbound['id']
   $client_obj: associative array representing client fields (id, email, subId, enable, expiryTime, totalGB, etc)
   Returns decoded server response (array) or null on failure.
*/
function panel_update_client_by_uuid($panel, $client_uuid, $inbound_id, $client_obj) {
    $dir = ensure_logs_dir();
    $cookiefile = panel_cookiefile($panel['id']);
    if (!file_exists($cookiefile) || filesize($cookiefile) < 10) {
        $ok = panel_login($panel);
        panel_debug_log($panel['id'], "panel_update: login result = " . ($ok ? 'true' : 'false'));
        if (!$ok) return null;
    }

    $urlBase = rtrim($panel['url'], '/');
    $endpoint = $urlBase . "/panel/api/inbounds/updateClient/" . urlencode($client_uuid);
    $payload = [
        'id' => intval($inbound_id),
        'settings' => json_encode(['clients' => [$client_obj]])
    ];

    $ch = curl_init($endpoint);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    $body = json_encode($payload);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json', 'Accept: application/json']);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $cookiefile);
    curl_setopt($ch, CURLOPT_COOKIEJAR, $cookiefile);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    $out = curl_exec($ch);
    $err = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    panel_debug_log($panel['id'], "UPDATE client endpoint {$endpoint} HTTP={$http} curl_err=" . ($err ?: 'none') . " resp_preview=" . substr($out ?? '', 0, 800));
    @file_put_contents($dir . "/panel_{$panel['id']}_update_resp.json", $out);

    if ($err) return null;
    if (!$out || $http < 200 || $http >= 400) return null;

    $decoded = @json_decode($out, true);
    if (!$decoded) return null;

    // Now reset the client's traffic after updating
    $reset_endpoint = $urlBase . "/panel/api/inbounds/{$inbound_id}/resetClientTraffic/" . urlencode($client_obj['email']);
    $ch = curl_init($reset_endpoint);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Accept: application/json']);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $cookiefile);
    curl_setopt($ch, CURLOPT_COOKIEJAR, $cookiefile);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    $reset_out = curl_exec($ch);
    $reset_err = curl_error($ch);
    $reset_http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    panel_debug_log($panel['id'], "RESET traffic endpoint {$reset_endpoint} HTTP={$reset_http} curl_err=" . ($reset_err ?: 'none') . " resp_preview=" . substr($reset_out ?? '', 0, 400));

    if ($reset_err || !$reset_out || $reset_http < 200 || $reset_http >= 400) {
        panel_debug_log($panel['id'], "Traffic reset failed for client {$client_uuid}");
    }

    return $decoded;
}

/* Utility: compute new expiryTime (ms) given existing expiryTime (ms) and add_hours */
function compute_new_expiry_ms($existing_expiry_ms, $add_hours) {
    $now_ms = (int)(microtime(true) * 1000);
    $add_ms = intval($add_hours) * 3600 * 1000;
    if ($existing_expiry_ms && intval($existing_expiry_ms) > 0) {
        return intval($existing_expiry_ms) + $add_ms;
    } else {
        return $now_ms + $add_ms;
    }
}

/* Convert bytes to GB (float) - used for totalGB field */
function bytes_to_gb($bytes) {
    if ($bytes <= 0) return 0;
    return round($bytes / (1024 * 1024 * 1024), 6); // keep precision
}

/* Format bytes human */
function format_bytes($b){
    if($b>=1073741824) return round($b/1073741824,2).' GB';
    if($b>=1048576) return round($b/1048576,2).' MB';
    if($b>=1024) return round($b/1024,2).' KB';
    return $b.' B';
}


/**
 * Fetch a Telegram chat/user display name via getChat.
 * Returns a readable name (first + last) or username if available, or empty string.
 */
function fetch_telegram_name($telegram_id) {
    global $config;
    if (empty($telegram_id) || empty($config['telegram_token'])) return '';
    $url = "https://api.telegram.org/bot{$config['telegram_token']}/getChat";
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, ['chat_id' => $telegram_id]);
    $res = curl_exec($ch);
    curl_close($ch);
    if (!$res) return '';
    $json = json_decode($res, true);
    if (!is_array($json) || empty($json['ok'])) return '';
    $chat = $json['result'] ?? [];
    $nameParts = [];
    if (!empty($chat['first_name'])) $nameParts[] = $chat['first_name'];
    if (!empty($chat['last_name'])) $nameParts[] = $chat['last_name'];
    if (!empty($nameParts)) return trim(implode(' ', $nameParts));
    if (!empty($chat['username'])) return $chat['username'];
    return '';
}
